(*----------------------------------------------------------------------(C)-*)
(* Copyright (C) 2006-2016 Konstantin Korovin and The University of Manchester. 
   This file is part of iProver - a theorem prover for first-order logic.

   iProver is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 2 of the License, or 
   (at your option) any later version.
   iProver is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
   See the GNU General Public License for more details.
   You should have received a copy of the GNU General Public License
   along with iProver.  If not, see <http://www.gnu.org/licenses/>.         *)
(*----------------------------------------------------------------------[C]-*)



open Lib

type stat_int_entry = {
  int_entry_name : string;
  mutable value  : int;
}

type stat_float_entry = {
  float_entry_name    : string;
  mutable float_value : float;
  (* mutable time_cache : float; *)
}

(* statistics for stat_fun_entry is generated by calling stat_fun *)
(* stat_fun can be assigned by any module *)
type stat_fun_entry = {
  fun_entry_name          : string;
  mutable stat_fun        : (unit -> int) param;
}

let make_int ?(init=0) name = {
  int_entry_name = name;
  value = init;
}

let make_float ?(init=0.) name = {
  float_entry_name = name;
  float_value = init;
  (* time_cache = 0.; *)
}

let make_fun name = {
  fun_entry_name = name;
  stat_fun = Undef;
}



(*-----------------*)
let assign_int_stat value stat  = 
  stat.value <- value

let bump_int_stat stat = 
  stat.value <- succ stat.value 

let incr_int_stat int stat = 
  stat.value <- stat.value + int

let get_val_stat s = s.value

let get_float_val_stat s = s.float_value

let assign_float_stat value stat = 
  stat.float_value <- value

let add_float_stat float stat = 
  stat.float_value <- (stat.float_value +. float)

let get_val_stat_fun stat_fun_entry = 
  match stat_fun_entry.stat_fun with
  | Def f -> f ()
  | Undef -> failwith 
    (sprintf "Statistics for function %s is undefined" (stat_fun_entry.fun_entry_name))

let get_val_stat_fun0 stat_fun_entry = 
  match stat_fun_entry.stat_fun with
  | Def f -> f ()
  | Undef -> 0
              
        
let assign_fun_stat f stat  = 
  stat.stat_fun <- Def (f)

let clear_stat_int_entry e = e.value <- 0
let clear_stat_fun_entry e = e.stat_fun <- Undef


(* does not work! *)
(* let run_and_time stat f x =
  let start_time = Unix.gettimeofday () in
  (* out_str ("s "^(string_of_float start_time)^"\n");*)
  try
    let res = f x in
    let end_time   = Unix.gettimeofday () in 
    (* out_str ("e "^(string_of_float end_time)^"\n");*)
    let run_time = (end_time -. start_time) in
    (* out_str ("r "^(string_of_float run_time)^"\n");*)
    add_float_stat run_time stat;
    res
  with e -> 
    let end_time   = Unix.gettimeofday () in 
    let run_time = (end_time -. start_time) in
    out_str "adding\n";
    add_float_stat run_time stat;      
    raise e *)



(* Time statistics *)

let time_global_flag = true

let[@inline] time (stat: stat_float_entry) f = 
  if time_global_flag then (
    let t_start = Unix.gettimeofday() in
    let record_time () =    
      let t_end = Unix.gettimeofday() in
      stat.float_value <- stat.float_value +. t_end -. t_start;
    in  
    try    
      let res = f () in 
      record_time ();
      res
    with x -> 
      record_time ();
      raise_trace x
   )
  else (f ())


(* deprecated *)
(* note that if several starts the latest rewrites stat.time_cache *)
(* also if from the block exit by exception before time_end; stat.time_cache remains and only will be rewrittent next time_start *)
(*
let time_start (stat: stat_float_entry) = 
  if time_global_flag then (
    let t_start = Unix.gettimeofday() in
    stat.time_cache <- t_start   
  )

let time_end (stat: stat_float_entry) = 
  if time_global_flag then (
    let t_start = stat.time_cache in
    let t_end = Unix.gettimeofday() in
    stat.float_value <- stat.float_value +. t_end -. t_start;
    stat.time_cache <- 0. (* in order to output stats on interruptions *)
  )
*)


(*------- Problem properties---------*)

let clauses = 
  make_int "clauses"

let conjectures = 
  make_int "conjectures"

let epr = 
  make_int "epr"
    
let horn = 
  make_int "horn"

let ground = 
  make_int "ground"

let unary = 
  make_int "unary"

let binary = 
  make_int "binary"

let lits = 
  make_int "lits"
    
let lits_eq = 
  make_int "lits_eq"

let fd_pure = 
  make_int "fd_pure"

let fd_pseudo =
  make_int "fd_pseudo"

let fd_cond = 
  make_int "fd_cond"
    
let fd_pseudo_cond = 
  make_int "fd_pseudo_cond"

let ac_symbols = 
  make_fun "ac_symbols"


(*-------General---------*)

let abstr_ref_over_cycles =
  make_int "abstr_ref_over_cycles"

let abstr_ref_under_cycles =
  make_int "abstr_ref_under_cycles"

let gc_basic_clause_elim =
  make_int "gc_basic_clause_elim"

let num_of_symbols = 
  make_fun "num_of_symbols"

let num_of_terms = 
  make_fun "num_of_terms"

(*-- timings *)
let parsing_time = 
  make_float "parsing_time"

let unif_index_cands_time = 
  make_float "unif_index_cands_time"
 
let unif_index_add_time = 
  make_float "unif_index_add_time"

let orderings_time = 
  make_float "orderings_time"

let out_proof_time = 
  make_float "out_proof_time"

let total_time = 
  make_float "total_time"


(* used but is not ouput *)
let num_of_input_clauses = 
  make_int "num_of_input_clauses"

(* used but is not ouput *)
let num_of_input_neg_conjectures = 
  make_int "num_of_input_neg_conjectures"

(*-------- Preprocessing ---------*)

let num_of_splits = 
  make_int "num_of_splits"

let num_of_fresh_symb = 
  make_int "num_of_fresh_symb"

let num_of_reused_defs = 
  make_int "num_of_reused_defs"

let num_eq_ax_congr_red = 
  make_int "num_eq_ax_congr_red"

let num_of_sem_filtered_clauses = 
  make_int "num_of_sem_filtered_clauses"

(* non-collapsed subtypes *)
let num_of_subtypes = 
  make_int "num_of_subtypes"

let monotx_restored_types = 
  make_int "monotx_restored_types"

let sat_num_of_epr_types = 
  make_int "sat_num_of_epr_types"

let sat_num_of_non_cyclic_types = 
  make_int "sat_num_of_non_cyclic_types"

let sat_num_of_guarded_non_collapsed_types = 
  make_int "sat_guarded_non_collapsed_types"

let pure_diseq_elim = 
  make_int "num_pure_diseq_elim"

let simp_replaced_by =
  make_int "simp_replaced_by"

let res_preprocessed =
  make_int "res_preprocessed"

let sup_preprocessed =
  make_int "sup_preprocessed"

let prep_upred =
  make_int "prep_upred"

let prep_unflattend = 
  make_int "prep_unflattend"

let prep_well_definedness =
  make_int "prep_well_definedness"

let smt_new_axioms = 
  make_int "smt_new_axioms"

let pred_elim_cands = 
  make_int "pred_elim_cands"

let pred_elim = 
  make_int "pred_elim"

let pred_elim_cl = 
  make_int "pred_elim_cl"

let pred_elim_cycles = 
  make_int "pred_elim_cycles"

let merged_defs = 
  make_int "merged_defs"

let merged_defs_ncl = 
  make_int "merged_defs_ncl"

let bin_hyper_res = 
  make_int "bin_hyper_res"

let prep_cycles = 
  make_int "prep_cycles"

(*-- timings *)
let splitting_time = 
  make_float "splitting_time"

let sem_filter_time = 
  make_float "sem_filter_time"

let monotx_time = 
  make_float "monotx_time"
 
let subtype_inf_time = 
  make_float "subtype_inf_time"
 
let res_prep_time =
  make_float "res_prep_time"

let sup_prep_time =
  make_float "sup_prep_time"

let pred_elim_time = 
  make_float "pred_elim_time"

let pred_elim_time = 
  make_float "pred_elim_time"
   
let bin_hyper_res_time = 
   make_float "bin_hyper_res_time"

let prep_time_total =
  make_float "prep_time_total"
 
(*--------Propositonal Solver---------*)

let prop_solver_calls = 
  make_fun "prop_solver_calls"

let prop_fast_solver_calls = 
  make_fun "prop_fast_solver_calls"

let smt_solver_calls = 
  make_int "smt_solver_calls"

let smt_fast_solver_calls = 
  make_int "smt_fast_solver_calls"

let prop_num_of_clauses = 
  make_int "prop_num_of_clauses"

let prop_preprocess_simplified = 
  make_int "prop_preprocess_simplified"

let prop_fo_subsumed = 
  make_int "prop_fo_subsumed"

(*-- timings *)
let prop_solver_time = 
  make_float "prop_solver_time"

let prop_fast_solver_time = 
  make_float "prop_fast_solver_time"

let prop_unsat_core_time = 
  make_float "prop_unsat_core_time"

let smt_solver_time = 
  make_float "smt_solver_time"

let smt_fast_solver_time = 
  make_float "smt_fast_solver_time"

(*------- QBF ----------*)

let qbf_q_res = 
  make_int "qbf_q_res"

let qbf_num_tautologies =
  make_int "qbf_num_tautologies"

let qbf_prep_cycles = 
  make_int "qbf_prep_cycles"


(*------BMC1- --------------------------*)
let bmc1_current_bound =
  make_int ~init:(-1) "bmc1_current_bound"

let bmc1_last_solved_bound = 
  make_int ~init:(-1) "bmc1_last_solved_bound"

let bmc1_unsat_core_size = 
  make_int ~init:(-1) "bmc1_unsat_core_size"

let bmc1_unsat_core_parents_size = 
  make_int ~init:(-1) "bmc1_unsat_core_parents_size"

let bmc1_merge_next_func = 
  make_int "bmc1_merge_next_fun"

let bmc1_unsat_core_clauses_time = 
  make_float "bmc1_unsat_core_clauses_time"


(*------ Instantiation --------------*)

let inst_num_of_clauses = 
  make_fun "inst_num_of_clauses"

let inst_num_in_passive = 
  make_fun "inst_num_in_passive"

let inst_num_in_active = 
  make_int "inst_num_in_active"

let inst_num_in_unprocessed  = 
  make_int "inst_num_in_unprocessed"
 
let inst_num_of_loops = 
  make_int "inst_num_of_loops"

let inst_num_of_learning_restarts  = 
  make_int "inst_num_of_learning_restarts"
   
let inst_num_moves_active_passive = 
  make_int "inst_num_moves_active_passive"

let inst_max_lit_activity  = 
  make_int "inst_lit_activity"

let inst_lit_activity_moves  = 
  make_int "inst_lit_activity_moves" 

let inst_num_tautologies  = 
  make_int "inst_num_tautologies" 

let inst_num_prop_implied  = 
  make_int "inst_num_prop_implied"

let inst_num_existing_simplified  = 
  make_int "inst_num_existing_simplified" 

let inst_num_eq_res_simplified =
  make_int "inst_num_eq_res_simplified" 

let inst_num_child_elim = 
  make_int "inst_num_child_elim"

let inst_num_of_dismatching_blockings  = 
  make_int "inst_num_of_dismatching_blockings"

let inst_num_of_non_proper_insts  = 
  make_int "inst_num_of_non_proper_insts"

let inst_num_of_duplicates  = 
  make_int "inst_num_of_duplicates"

let inst_num_from_inst_to_res  = 
  make_int "inst_inst_num_from_inst_to_res" 

(*-- timings *)

let inst_time_sim_new = 
  make_float "inst_time_sim_new"

let inst_time_sim_given = 
  make_float "inst_time_sim_given"

let inst_time_dismatching_checking = 
  make_float "inst_time_dismatching_checking"

let inst_time_total = 
  make_float "inst_time_total"
    

(*------Resolution------------------*)

let res_num_of_clauses = 
  make_fun "res_num_of_clauses"

let res_num_in_passive = 
  make_fun "res_num_in_passive"

let res_num_in_active = 
  make_int "res_num_in_active"

let res_num_of_loops = 
  make_int "res_num_of_loops"

let res_forward_subset_subsumed  = 
  make_int "res_forward_subset_subsumed"

let res_backward_subset_subsumed = 
  make_int "res_backward_subset_subsumed"

let res_forward_subsumed = 
  make_int "res_forward_subsumed"

let res_backward_subsumed = 
  make_int "res_backward_subsumed"

let res_forward_subsumption_resolution = 
  make_int "res_forward_subsumption_resolution"

let res_backward_subsumption_resolution = 
  make_int "res_backward_subsumption_resolution"
    
let res_clause_to_clause_subsumption = 
  make_int "res_clause_to_clause_subsumption"

let res_subs_bck_cnt = 
  make_int "res_subs_bck_cnt" 

let res_orphan_elimination = 
  make_int "res_orphan_elimination"

let res_tautology_del = 
  make_int "res_tautology_del" 

let res_num_eq_res_simplified =
  make_int "res_num_eq_res_simplified" 

let res_num_sel_changes = 
  make_int "res_num_sel_changes"

let res_moves_from_active_to_pass = 
  make_int "res_moves_from_active_to_pass"

(*-- timings *)


let res_time_sim_new = 
  make_float "res_time_sim_new"

let res_time_sim_fw_given = 
  make_float "res_time_sim_fw_given"

let res_time_sim_bw_given = 
  make_float "res_time_sim_bw_given"

let res_time_total = 
  make_float "res_time_total"
  

(*------ Superposition ------*)

let sup_num_of_clauses = 
  make_fun "sup_num_of_clauses"

let sup_num_in_active = 
  make_fun "sup_num_in_active"

let sup_num_in_passive = 
  make_fun "sup_num_in_passive"

let sup_num_of_loops = 
  make_int "sup_num_of_loops"

let sup_fw_superposition = 
  make_int "sup_fw_superposition"

let sup_bw_superposition = 
  make_int "sup_bw_superposition"

let sup_eq_factoring = 
  make_int "sup_eq_factoring"

let sup_eq_resolution = 
  make_int "sup_eq_resolution"

let sup_given_eliminated = 
  make_int "sup_given_eliminated"

let sup_immediate_simplified = 
  make_int "sup_immediate_simplified"

let comparisons_done = 
  make_int "comparisons_done"

let comparisons_avoided = 
  make_int "comparisons_avoided"

let comparisons_inc_criteria = 
  make_int "comparisons_inc_criteria"

let sup_deep_cl_discarded = 
  make_int "sup_deep_cl_discarded"

let sup_num_of_deepenings = 
  make_int "sup_num_of_deepenings"

let sup_num_of_restarts = 
  make_int "sup_num_of_restarts"

(*-- timings *)

let sup_time_generating = 
  make_float "sup_time_generating"

(*
let sup_time_sim_full = 
  make_float "sup_time_sim_full"
*)
let sup_time_sim_fw_full = 
  make_float "sup_time_sim_fw_full"

let sup_time_sim_bw_full = 
  make_float "sup_time_sim_bw_full"

let sup_time_sim_fw_immed = 
  make_float "sup_time_sim_fw_immed"

let sup_time_sim_bw_immed = 
  make_float "sup_time_sim_bw_immed"

let sup_time_prep_sim_fw_input = 
  make_float "sup_time_prep_sim_fw_input"

let sup_time_prep_sim_bw_input = 
  make_float "sup_time_prep_sim_bw_input"

let sup_time_total = 
  make_float "sup_time_total"

(*
let sup_forward_subset_subsumed = 
  make_int "sup_forward_subset_subsumed"

let sup_forward_subsumed = 
  make_int "sup_forward_subsumed"

let sup_forward_subsumption_resolution = 
  make_int "sup_forward_subsumption_resolution"

let sup_clause_to_clause_subsumption = 
  make_int "sup_clause_to_clause_subsumption"

let sup_tautology_del = 
  make_int "sup_tautology_del"

let sup_fw_demodulated = 
  make_int "sup_fw_demodulated"

let sup_bw_demodulated = 
  make_int "sup_bw_demodulated"

*)

(* --- Simplification --- *)
let sim_repeated = 
  make_int "sim_repeated"

let sim_fw_subset_subsumed = 
  make_int "sim_fw_subset_subsumed"

let sim_bw_subset_subsumed  = 
  make_int "sim_bw_subset_subsumed"

let sim_fw_subsumed  = 
  make_int "sim_fw_subsumed"

let sim_bw_subsumed = 
  make_int "sim_bw_subsumed"

let sim_fw_subsumption_res = 
  make_int "sim_fw_subsumption_res"

let sim_bw_subsumption_res  = 
  make_int "sim_bw_subsumption_res"

let sim_fw_unit_subs  = 
  make_int "sim_fw_unit_subs"

let sim_bw_unit_subs  = 
  make_int "sim_bw_unit_subs"

let sim_tautology_del = 
  make_int "sim_tautology_del"

let sim_eq_tautology_del = 
  make_int "sim_eq_tautology_del"

let sim_eq_res_simp = 
  make_int "sim_eq_res_simp"

let sim_fw_demodulated = 
  make_int "sim_fw_demodulated"

let sim_bw_demodulated = 
  make_int "sim_bw_demodulated"

let sim_encompassment_demod = 
  make_int "sim_encompassment_demod"

let sim_demod_cache_hit_pos =
  make_int "sim_demod_cache_hit_pos"

let sim_demod_cache_hit_neg =
  make_int "sim_demod_cache_hit_neg"

let sim_demod_cache_miss =
  make_int "sim_demod_cache_miss"
        
let sim_light_normalised = 
  make_int "sim_light_normalised"

let sim_ac_normalised = 
  make_int "sim_ac_normalised"

(* removed _ac_ in name to be backwards compatible with ML *)
let sim_ac_joinable_taut = 
  make_int "sim_joinable_taut" 
(*  make_int "sim_ac_joinable_taut" *)

(* removed _ac_ in name to be backwards compatible with ML *)
let sim_ac_joinable_simp = 
  make_int "sim_joinable_simp"
(*  make_int "sim_ac_joinable_simp"*)

let sim_smt_subsumption = 
  make_int "sim_smt_subsumption"

let sim_fw_ac_demod = 
  make_int "sim_fw_ac_demod"

let sim_bw_ac_demod = 
  make_int "sim_bw_ac_demod"

let sim_smt_simplified = 
  make_int "sim_smt_simplified"

let sim_ground_joinable = 
  make_int "sim_ground_joinable"

let sim_bw_ground_joinable = 
  make_int "sim_bw_ground_joinable"

let sim_connectedness = 
  make_int "sim_connectedness"

(*-- timing *)

let sim_time_fw_subset_subs = 
  make_float "sim_time_fw_subset_subs"

let sim_time_bw_subset_subs  = 
  make_float "sim_time_bw_subset_subs"

let sim_time_fw_subs  = 
  make_float "sim_time_fw_subs"

let sim_time_bw_subs = 
  make_float "sim_time_bw_subs"

let sim_time_fw_subs_res = 
  make_float "sim_time_fw_subs_res"

let sim_time_bw_subs_res  = 
  make_float "sim_time_bw_subs_res"

let sim_time_fw_unit_subs  = 
  make_float "sim_time_fw_unit_subs"

let sim_time_bw_unit_subs = 
  make_float "sim_time_bw_unit_subs"

let sim_time_tautology_del = 
  make_float "sim_time_tautology_del"

let sim_time_eq_tautology_del = 
  make_float "sim_time_eq_tautology_del"

let sim_time_eq_res_simp = 
  make_float "sim_time_eq_res_simp"

let sim_time_fw_demod = 
  make_float "sim_time_fw_demod"

let sim_time_bw_demod = 
  make_float "sim_time_bw_demod"

let sim_time_light_norm = 
  make_float "sim_time_light_norm"

(* counted in sim_time_joinable_sim/sim_time_ac_normalised *)
(* let sim_time_joinable_taut = 
  make_float "sim_time_joinable_taut"

let sim_time_joinable_simp = 
  make_float "sim_time_joinable_simp" *)

let sim_time_joinable = (* TODO change to ac joinable, confusing with ground joinability, but check first that this doesn't break anything *)
  make_float "sim_time_joinable"

let sim_time_ac_norm = 
  make_float "sim_time_ac_norm"

let sim_time_fw_ac_demod = 
  make_float "sim_time_fw_ac_demod"

let sim_time_bw_ac_demod = 
  make_float "sim_time_bw_ac_demod"

let sim_time_smt_subs = 
  make_float "sim_time_smt_subs"

let sim_time_fw_gjoin = 
  make_float "sim_time_fw_gjoin"

let sim_time_fw_connected = 
  make_float "sim_time_fw_connected"



(*-----------------*)

let val_dist = 40

let stat_int_to_stream stream s = 
  (* (space_padding_str val_dist ((tptp_safe_str s.int_entry_name)^":"))
  ^(string_of_int s.value) *)
  space_padding_stream val_dist stream (tptp_safe_str s.int_entry_name ^ ":");
  print_int s.value

(* truncates to 3 digits after .*)
let stat_float_to_stream stream s = 
  (* (space_padding_str val_dist ((tptp_safe_str s.float_entry_name)^":"))
  ^(string_of_float (truncate_n 3 s.float_value)) *)
  space_padding_stream val_dist stream (tptp_safe_str s.float_entry_name ^ ":");
  print_float @@ truncate_n 3 s.float_value

let stat_fun_to_stream stream s = 
  let val_str = 
    match s.stat_fun with 
    | Def f ->  string_of_int (f ())
    | Undef -> "undef"
  in
  space_padding_stream val_dist stream (tptp_safe_str s.fun_entry_name ^ ":");
  print_string val_str

let stat_int_list_stdout s_list = 
  list_to_stream stdout_stream stat_int_to_stream s_list "\n"

let stat_float_list_stdout s_list = 
  list_to_stream stdout_stream stat_float_to_stream s_list "\n"

let stat_fun_list_stdout s_list = 
  list_to_stream stdout_stream stat_fun_to_stream s_list "\n"



(* Print only nonzero *)

let nonzero_int_to_stream stream s = 
  if not (s.value == 0 || s.value = -1) then (stat_int_to_stream stream s; print_newline())

let nonzero_float_to_stream stream s = 
  if Float.O.(s.float_value <> 0.) then (stat_float_to_stream stream s; print_newline())

let nonzero_fun_to_stream stream s = 
  match s.stat_fun with 
  | Def f -> 
    let value = f () in 
    if value <> 0 then (
      let val_str = string_of_int value in
      space_padding_stream val_dist stream (tptp_safe_str s.fun_entry_name ^ ":");
      print_endline val_str
    )
  | Undef -> ()

let nonzero_int_list_stdout s_list = 
  List.iter (nonzero_int_to_stream stdout_stream) s_list

let nonzero_float_list_stdout s_list = 
  List.iter (nonzero_float_to_stream stdout_stream) s_list

let nonzero_fun_list_stdout s_list = 
  List.iter (nonzero_fun_to_stream stdout_stream) s_list



(*--------General--------*)

let gen_fun_stat_list =
  [
   num_of_symbols;
   num_of_terms;
 ]
    
let gen_int_stat_list =
  [
    abstr_ref_over_cycles;
    abstr_ref_under_cycles;
    gc_basic_clause_elim;
  ]

let gen_float_stat_list = 
  [
   parsing_time;
   unif_index_cands_time;
   unif_index_add_time;
   orderings_time;
   out_proof_time;
   total_time;
 ]    


(*---- propblem properites ------*)
let problem_props_int_stat_list = 
  [
   clauses;
   conjectures;
   epr;
   horn;
   ground;
   unary;
   binary;
   lits;
   lits_eq;
   fd_pure;
   fd_pseudo;
   fd_cond;
   fd_pseudo_cond;
 ]

let problem_props_fun_stat_list = [
   ac_symbols;
]

(*---- preprocessing ------*)

let prep_int_stat_list = 
  [
   num_of_splits;
   num_of_fresh_symb;
   num_of_reused_defs;
   num_eq_ax_congr_red;
   num_of_sem_filtered_clauses;
   num_of_subtypes;
   monotx_restored_types;
   sat_num_of_epr_types;
   sat_num_of_non_cyclic_types;
   sat_num_of_guarded_non_collapsed_types;
   pure_diseq_elim;
   simp_replaced_by;
   res_preprocessed;
   sup_preprocessed;
   prep_upred;
   prep_unflattend;
   prep_well_definedness;
   smt_new_axioms;
   pred_elim_cands;
   pred_elim;
   pred_elim_cl;
   pred_elim_cycles;
   merged_defs;
   merged_defs_ncl;
   bin_hyper_res;
   prep_cycles;
 ]

let prep_float_stat_list = 
  [
   splitting_time;
   sem_filter_time;
   monotx_time;
   subtype_inf_time;
   res_prep_time;
   sup_prep_time;
   pred_elim_time;
   bin_hyper_res_time;
   prep_time_total;
 ]

(*----prop solver-----*)
let prop_solver_fun_stat_list = 
  [
   prop_solver_calls;
   prop_fast_solver_calls;
 ]
    
let prop_solver_int_stat_list = 
  [
   smt_solver_calls;
   smt_fast_solver_calls;
   prop_num_of_clauses;
   prop_preprocess_simplified;
   prop_fo_subsumed;   
 ]

let prop_solver_float_stat_list = [
  prop_solver_time;
  prop_fast_solver_time;
  prop_unsat_core_time;
  smt_solver_time;
  smt_fast_solver_time;

]

(*-------qbf ---------*)
let qbf_int_stat_list = 
  [
   qbf_q_res;
   qbf_num_tautologies;
   qbf_prep_cycles;
 ]

(*---- bmc1 --------*)
let bmc1_int_stat_list =
  [ bmc1_current_bound;
    bmc1_last_solved_bound; 
    bmc1_unsat_core_size; 
    bmc1_unsat_core_parents_size;
    bmc1_merge_next_func
 ]

let bmc1_float_stat_list =
  [ bmc1_unsat_core_clauses_time ]

(*-------instantiation --------*)
let inst_fun_stat_list = 
  [
   inst_num_of_clauses;
   inst_num_in_passive;
 ]

let inst_int_stat_list = 
  [
   inst_num_in_active;
   inst_num_of_loops;
   inst_num_in_unprocessed;
   inst_num_of_learning_restarts;
   inst_num_moves_active_passive;
   inst_max_lit_activity;
   inst_lit_activity_moves;
   inst_num_tautologies;
   inst_num_prop_implied;
   inst_num_existing_simplified;
   inst_num_eq_res_simplified;
   inst_num_child_elim;
   inst_num_of_dismatching_blockings;
   inst_num_of_non_proper_insts;
   inst_num_of_duplicates;
   inst_num_from_inst_to_res;
 ]


let inst_float_stat_list = 
  [
   inst_time_sim_new;
   inst_time_sim_given;
   inst_time_dismatching_checking; 
   inst_time_total
 ]

(*----- resolution -------*)
let res_fun_stat_list = 
  [
   res_num_of_clauses;
   res_num_in_passive;
 ]

let res_int_stat_list = 
  [
   res_num_in_active;
   res_num_of_loops;
   res_forward_subset_subsumed;
   res_backward_subset_subsumed; 
   res_forward_subsumed; 
   res_backward_subsumed;
   res_forward_subsumption_resolution;
   res_backward_subsumption_resolution; 
   res_clause_to_clause_subsumption;
   res_subs_bck_cnt;
   res_orphan_elimination;
   res_tautology_del; 
   res_num_eq_res_simplified;
   res_num_sel_changes;
   res_moves_from_active_to_pass;
 ]

let res_float_stat_list = 
  [
   res_time_sim_new;
   res_time_sim_fw_given;
   res_time_sim_bw_given;
   res_time_total
 ]


(*----- superposition -------*)
let sup_float_stat_list = [
  sup_time_generating;
  sup_time_sim_fw_full;
  sup_time_sim_bw_full;
  sup_time_sim_fw_immed;
  sup_time_sim_bw_immed;
  sup_time_prep_sim_fw_input;
  sup_time_prep_sim_bw_input;
  sup_time_total;
]

let sup_fun_stat_list = [
  sup_num_of_clauses;
  sup_num_in_active;
  sup_num_in_passive;
]

let sup_int_stat_list = [
  sup_num_of_loops;
  (* sup_forward_subset_subsumed; *)
  (* sup_backward_subset_subsumed;  *)
  (* sup_forward_subsumed;  *)
  (* sup_backward_subsumed; *)
  (* sup_forward_subsumption_resolution; *)
  (* sup_backward_subsumption_resolution;  *)
  (* sup_clause_to_clause_subsumption; *)
  (* sup_orphan_elimination; *)
  (* sup_tautology_del;  *)
  (* sup_num_eq_sup_simplified; *)
  (* sup_fw_demodulated;
  sup_bw_demodulated; *)

  sup_fw_superposition;
  sup_bw_superposition;
  sup_eq_factoring; 
  sup_eq_resolution; 

  sup_immediate_simplified;
  sup_given_eliminated;

  comparisons_done;
  comparisons_avoided;
  comparisons_inc_criteria;

  sup_deep_cl_discarded;
  sup_num_of_deepenings;
  sup_num_of_restarts;
]

(*----- simplifications -------*)
let sim_fun_stat_list = [
]

let sim_int_stat_list = [
  sim_repeated;
  sim_fw_subset_subsumed;
  sim_bw_subset_subsumed; 
  sim_fw_subsumed; 
  sim_bw_subsumed;
  sim_fw_subsumption_res;
  sim_bw_subsumption_res; 
  sim_fw_unit_subs;
  sim_bw_unit_subs;
  sim_tautology_del;
  sim_eq_tautology_del;
  sim_eq_res_simp;

  sim_fw_demodulated;
  sim_bw_demodulated;
  sim_encompassment_demod;
  sim_demod_cache_hit_pos;
  sim_demod_cache_hit_neg;
  sim_demod_cache_miss;
  
  sim_light_normalised;
  sim_ac_normalised;
  sim_ac_joinable_taut;
  sim_ac_joinable_simp;
(* removed _ac_demod be backwards compatible with ML *)

  sim_fw_ac_demod;
  sim_bw_ac_demod;

  sim_smt_subsumption;
  sim_smt_simplified;
  
  sim_ground_joinable;
  sim_bw_ground_joinable;
  sim_connectedness;
]

let sim_float_stat_list = [  
  sim_time_fw_subset_subs;
  sim_time_bw_subset_subs; 
  sim_time_fw_subs; 
  sim_time_bw_subs;
  sim_time_fw_subs_res;
  sim_time_bw_subs_res; 
  sim_time_fw_unit_subs;
  sim_time_bw_unit_subs;
  sim_time_tautology_del;
  sim_time_eq_tautology_del;
  sim_time_eq_res_simp;

  sim_time_fw_demod;
  sim_time_bw_demod;

  sim_time_light_norm;
  (* sim_time_joinable_taut; *)
  (* sim_time_joinable_simp; *)
  sim_time_joinable;
  sim_time_ac_norm;
(* removed _ac_demod be backwards compatible with ML *)

  sim_time_fw_ac_demod;
  sim_time_bw_ac_demod;

  sim_time_smt_subs;
  sim_time_fw_gjoin;
  sim_time_fw_connected;
]


let clear_inst_stat () = 
  List.iter clear_stat_int_entry inst_int_stat_list;
  List.iter clear_stat_fun_entry inst_fun_stat_list

let clear_res_stat () = 
  List.iter clear_stat_int_entry res_int_stat_list;
  List.iter clear_stat_fun_entry res_fun_stat_list

let clear_sup_stat () = 
  List.iter clear_stat_int_entry sup_int_stat_list;
  List.iter clear_stat_fun_entry sup_fun_stat_list

let stat_to_stdout () = 
  space_padding_stream (val_dist-5) stdout_stream (s_pref_str());
  print_string "  Statistics\n\n"; 

(*  printf "%s%s" (s_pref_str()) "Statistics\n\n"; *)
  printf "%s%s" (s_pref_str()) "Problem properties\n\n";
  (stat_int_list_stdout problem_props_int_stat_list); print_string "\n";
  (stat_fun_list_stdout problem_props_fun_stat_list); print_string "\n\n";
  
  printf "%s%s" (s_pref_str()) "General\n\n";
  (stat_int_list_stdout  gen_int_stat_list); print_string "\n";
  (stat_fun_list_stdout gen_fun_stat_list); print_string "\n\n";
  (stat_float_list_stdout gen_float_stat_list); print_string "\n\n";

  printf "%s%s" (s_pref_str()) "Preprocessing\n\n";
  (stat_int_list_stdout prep_int_stat_list); print_string "\n\n";
  (stat_float_list_stdout prep_float_stat_list); print_string "\n\n";

  printf "%s%s" (s_pref_str()) "Propositional Solver\n\n";
  (stat_fun_list_stdout prop_solver_fun_stat_list); print_string "\n";
  (stat_int_list_stdout prop_solver_int_stat_list); print_string "\n\n";
  (stat_float_list_stdout prop_solver_float_stat_list); print_string "\n\n";

  printf "%s%s" (s_pref_str()) "QBF\n\n";
  (stat_int_list_stdout qbf_int_stat_list); print_string "\n\n";

  printf "%s%s" (s_pref_str()) "BMC1\n\n";
  (stat_int_list_stdout bmc1_int_stat_list); print_string "\n\n";
  (stat_float_list_stdout bmc1_float_stat_list); print_string "\n\n";

  printf "%s%s" (s_pref_str()) "Instantiation\n\n";
  (stat_fun_list_stdout inst_fun_stat_list); print_string "\n";
  (stat_int_list_stdout inst_int_stat_list); print_string "\n\n";
  (stat_float_list_stdout inst_float_stat_list); print_string "\n\n";

  printf "%s%s" (s_pref_str()) "Resolution\n\n";
  (stat_fun_list_stdout res_fun_stat_list); print_string "\n";
  (stat_int_list_stdout res_int_stat_list); print_string "\n\n";
  (stat_float_list_stdout res_float_stat_list); print_string "\n\n";


  printf "%s%s" (s_pref_str()) "Superposition\n\n";
  (stat_fun_list_stdout sup_fun_stat_list); print_string "\n";
  (stat_int_list_stdout sup_int_stat_list); print_string "\n\n";
  (stat_float_list_stdout sup_float_stat_list); print_string "\n\n";

  printf "%s%s" (s_pref_str()) "Simplifications\n\n";
  if (sim_fun_stat_list != []) then ((stat_fun_list_stdout sim_fun_stat_list); print_string "\n");
  (stat_int_list_stdout sim_int_stat_list); print_string "\n\n";
  (stat_float_list_stdout sim_float_stat_list); print_string "\n\n";

  ()

let nonzero_to_stdout () = 
  space_padding_stream (val_dist-5) stdout_stream (s_pref_str());
  print_string "  Statistics\n\n";
  
  printf "%s%s" (s_pref_str()) "General\n\n";
  (nonzero_int_list_stdout gen_int_stat_list);
  (nonzero_float_list_stdout gen_float_stat_list);
  (nonzero_fun_list_stdout gen_fun_stat_list); print_string "\n";

  printf "%s%s" (s_pref_str()) "Preprocessing\n\n";
  (nonzero_int_list_stdout prep_int_stat_list);
  (nonzero_float_list_stdout prep_float_stat_list); print_string "\n";

  printf "%s%s" (s_pref_str()) "Problem properties\n\n";
  (nonzero_int_list_stdout problem_props_int_stat_list);
  (nonzero_fun_list_stdout problem_props_fun_stat_list); print_string "\n";

  printf "%s%s" (s_pref_str()) "Propositional Solver\n\n";
  (nonzero_fun_list_stdout prop_solver_fun_stat_list);
  (nonzero_int_list_stdout prop_solver_int_stat_list);
  (nonzero_float_list_stdout prop_solver_float_stat_list); print_string "\n";

  printf "%s%s" (s_pref_str()) "QBF\n\n";
  (nonzero_int_list_stdout qbf_int_stat_list); print_string "\n";

  printf "%s%s" (s_pref_str()) "BMC1\n\n";
  (nonzero_int_list_stdout bmc1_int_stat_list);
  (nonzero_float_list_stdout bmc1_float_stat_list); print_string "\n";

  printf "%s%s" (s_pref_str()) "Instantiation\n\n";
  (nonzero_fun_list_stdout inst_fun_stat_list);
  (nonzero_int_list_stdout inst_int_stat_list);
  (nonzero_float_list_stdout inst_float_stat_list); print_string "\n";

  printf "%s%s" (s_pref_str()) "Resolution\n\n";
  (nonzero_fun_list_stdout res_fun_stat_list);
  (nonzero_int_list_stdout res_int_stat_list); print_string "\n";

  printf "%s%s" (s_pref_str()) "Superposition\n\n";
  (nonzero_float_list_stdout sup_float_stat_list);
  (nonzero_fun_list_stdout sup_fun_stat_list);
  (nonzero_int_list_stdout sup_int_stat_list); print_string "\n";

  printf "%s%s" (s_pref_str()) "Simplifications\n\n";
  (* (nonzero_fun_list_stdout sim_fun_stat_list); *)
  (nonzero_int_list_stdout sim_int_stat_list);

  ()



(*---- selected statistics ----*)

let sel_float_stat_list = 
  [
   total_time;
   (*
   out_proof_time;
   prep_time_total;
   prop_solver_time;
     smt_solver_time;
    *)
   
   inst_time_total;
   res_time_total;    
   sup_time_total;
   sup_time_generating;
   sup_time_sim_fw_full;
   sup_time_sim_bw_full;
   sup_time_sim_fw_immed;
   sup_time_sim_bw_immed;

   sim_time_fw_subs;
   sim_time_bw_subs;
   sim_time_fw_subs_res;
   sim_time_bw_subs_res;
   sim_time_fw_unit_subs;
   sim_time_bw_unit_subs;
   sim_time_fw_demod;
   sim_time_bw_demod;
   sim_time_light_norm;
   sim_time_joinable;
   sim_time_ac_norm;
   sim_time_fw_ac_demod;
   sim_time_bw_ac_demod;
   sim_time_smt_subs;
   sim_time_fw_gjoin;   
   sim_time_fw_connected;
   
 ]    

let sel_int_stat_list =
  [
   sim_fw_unit_subs;
   sim_bw_unit_subs;
   sim_fw_subsumed;
   sim_bw_subsumed;
   sim_fw_demodulated;
   sim_bw_demodulated;
   sim_encompassment_demod;
   sim_demod_cache_hit_pos;
   sim_demod_cache_hit_neg;
   sim_demod_cache_miss;
   sim_light_normalised;
   sim_ac_normalised;
   sim_connectedness;   
 ]
    
let sel_stat_to_stdout () = 
  space_padding_stream (val_dist-5) stdout_stream pref_str;
  print_string "  Statistics\n\n";
  printf "%s%s" (s_pref_str()) "Selected\n\n";
  stat_int_list_stdout sel_int_stat_list; print_string "\n";
  stat_float_list_stdout sel_float_stat_list; print_string "\n"

let out_stat () = 
(* assign iprover running time *)
  let end_time   = Unix.gettimeofday () in 
  let iprover_total_time = truncate_n 3 (end_time -. iprover_start_time) in 
  assign_float_stat iprover_total_time total_time;
  match !Options.global_options.stats_out with 
  | Options.Stats_out_none -> ()
  | Options.Stats_out_sel -> sel_stat_to_stdout ()
  | Options.Stats_out_all -> stat_to_stdout ()
  | Options.Stats_out_nonzero -> nonzero_to_stdout ()
(*
  if (!Options.current_options.stats_out) 
  then 
    let end_time   = Unix.gettimeofday () in 
    let iprover_total_time = truncate_n 3 (end_time -. iprover_start_time) in 
    assign_float_stat iprover_total_time total_time;
    out_str (stat_to_str ())
  else
    ()
*)


(*

  type statistics = 
  {

(*---------General--------*)


  (* let lit_activity_check_move  = ref 0*)


  let inst_solver_thershold   = ref 100

 *)
