(* part of iProver
 *)

(* module for dealing with associativity functions *)

open Lib
open Logic_interface

  
(*----- debug modifiable part-----*)

let dbg_flag = true

type dbg_gr = 
  | D_trace
  | D_flat_ass

let dbg_gr_to_str = function 
  | D_trace -> "trace"
  | D_flat_ass -> "flat_ass"

let dbg_groups = [
  D_trace;
  D_flat_ass;
]
    
let module_name = __MODULE__

(*----- debug fixed part --------*)

let () = Lib.dbg_flag_msg dbg_flag module_name

let dbg group str_lazy =
  Lib.dbg_out_pref dbg_flag dbg_groups group dbg_gr_to_str module_name str_lazy

let dbg_env group f =
  Lib.dbg_env_set dbg_flag dbg_groups group f

(*----- debug -----*)


(* aplicable to ass and non-ass symb (g, term_list)
 *)
 
type flat_term =
    FT of symbol * (flat_term list) |
    FV of term (* var term *)  
 
let ft_to_string flat_term =
  let buf = Buffer.create 30 in
  
  let rec f ft =
    match ft with 
    | FT(symb, arg_list) ->
        Buffer.add_string buf (sprintf "[%s | " (Symbol.to_string symb));        
        f_arg arg_list; 
    |FV t ->
        Buffer.add_string buf (Term.to_string t);
  and
      f_arg arg_list =
    match arg_list with
    | h1::(h2::tl) ->
        (f h1);
        Buffer.add_string buf ",";
        f_arg (h2::tl)
    | [h1] ->
        (f h1);
        Buffer.add_string buf "]";
    | [] ->  Buffer.add_string buf "]";
  in
  f flat_term;
  Buffer.contents buf
    

type flat_env =
    {
     is_ass_symb : symbol -> bool;
       mutable term_to_flat : flat_term TMap.t;
   }

let create_env ~is_ass_symb =
  {
   is_ass_symb;
   term_to_flat = TMap.empty;
 }

let glb_env = ref None
    
let init_glb_env ~is_ass_symb =
  match !glb_env with
  | Some(env)-> ()
  | None -> glb_env := Some(create_env ~is_ass_symb)
  
    
(* Ex:

   merge_flat_list  f  [(f, l1]; (f,l2); (g,l3); (g,l4);  (f, l5); (f,l6)] =>
   (f,
        l1@l2@[(g,l3);g,l4]@l5@l6]) 

   A(g,l3)l2@l1
 *)

      
let merge_flat_list top_ass_symb ft_list =
  let list = 
    List.fold_left
      (fun acc ft ->
        match ft with
        | FT(symb, ft_arg_list) ->
            let new_acc = 
            if top_ass_symb == symb then
              ((List.rev ft_arg_list)@acc)
            else
              (ft::acc)
            in
            new_acc
        | FV t -> ft::acc
      )
      [] ft_list
  in
  FT(top_ass_symb, (List.rev list)) 
(*  FT(top_ass_symb, list) *)

    
(* ass_symb -- Some (current ass symbol) or None
   acc accumulated terms under current ass_synbol
*)

    
let rec flat_term' env acc term = 
  try
    TMap.find term env.term_to_flat
  with
    Not_found ->
      match term with        
      |Term.Fun (symb, args, _) ->
          let args = Term.arg_to_list args in
          let flat_args = List.map (flat_term' env acc) args in
          let ft_term =
            if env.is_ass_symb symb then
              merge_flat_list symb flat_args
            else
              FT(symb, flat_args)
          in
          
          env.term_to_flat <- TMap.add term ft_term env.term_to_flat;
          ft_term
      |Term.Var _ -> FV(term)


let flat_term env term =
  flat_term' env [] term 

     
(*-----*)
    
let test ~is_ass_symb term =
  let env = create_env ~is_ass_symb in
  let ft = flat_term env term in
(*  dbg D_trace @@ lazy (sprintf "flat: %s" (ft_to_string ft)); *)
  ft_to_string ft

        
(*  
  let ft = 
  dbg D_trace @@ lazy (sprintf "flat: %s" (ft_to_string ft));
       
*)


(*      
let test prob_prop term =
  let ass_symbs = Theory_db.get_associative_symbols (Theory_db.get_global_record ()) in
  let is_ass_symb symb = SSet.mem symb ass_symbs in
  let env = create_env is_ass_symb in
  let ft = flat_term env term in
  dbg D_trace @@ lazy (sprintf "flat: %s" (ft_to_string ft));
        ()
        *)
(*  
  let ft = 
  dbg D_trace @@ lazy (sprintf "flat: %s" (ft_to_string ft));
       
*)
