module ARI where

import Text.ParserCombinators.Parsec -- TODO: replace with megaparsec
import Data.Text as T
import qualified Data.Map as M

import Parsing

-- Scanners

comment :: Parser ()
comment = do
  _ <- char ';'
  skipMany (noneOf "\n")

whitespaces :: Parser ()
whitespaces = skipMany (do { _ <- space; return () } <|> comment)

simple_identifier :: Parser Text
simple_identifier = do
  whitespaces
  s <- many1 (noneOf "|(); \t\r\n")
  whitespaces
  return (T.pack s)

quoted_identifier :: Parser Text
quoted_identifier = do
  whitespaces
  _ <- char '|'
  s <- many1 (noneOf "| \t\r\n")
  _ <- char '|'
  whitespaces
  return (T.pack "|" <> T.pack s <> T.pack "|")

identifier :: Parser Text
identifier = try simple_identifier <|> quoted_identifier

number :: Parser Int
number = do
  whitespaces
  s <- many1 digit
  whitespaces
  return (read s :: Int)

keyword :: String -> Parser ()
keyword s = do
  whitespaces
  _ <- string s
  whitespaces

param :: String -> a -> Parser a
param s x = do
  keyword s
  return x
            
-- Parsing functions.

parse_format :: Parser ()
parse_format = do
  keyword "("
  keyword "format"
  keyword "TRS"
  keyword ")"

parse_fun :: Parser (Text, Int)
parse_fun = do
  keyword "("
  keyword "fun"
  f <- identifier
  n <- number
  keyword ")"
  return (f, n)

parse_term :: Signature -> Parser Term
parse_term sig = 
  try (parse_variable_or_constant sig) <|> 
  parse_function sig

parse_variable_or_constant :: Signature -> Parser Term
parse_variable_or_constant sig = do
  x <- identifier
  case M.lookup x sig of
    Nothing -> return (V x)
    Just 0  -> return (F x [])
    Just _  -> error (show x ++ " is not a constant.")

parse_function :: Signature -> Parser Term
parse_function sig = do
  keyword "("
  f <- identifier
  ts <- many (parse_term sig)
  keyword ")"
  case M.lookup f sig of
    Nothing            -> error (show f ++ " is not declared")
    Just n | m == n    -> return (F f ts)
           | m < n     -> error (show f ++ " takes too few arguemnts")
           | otherwise -> error (show f ++ " takes too many arguments")
      where m = Prelude.length ts

parse_rule :: Signature -> Parser (Term, Term)
parse_rule sig = do
  keyword "("
  keyword "rule"
  l <- parse_term sig
  r <- parse_term sig
  keyword ")"
  return (l, r)

parse_ari :: Parser (Signature, [(Term, Term)])
parse_ari = do
  parse_format
  sig' <- many (try parse_fun)
  let sig = M.fromList sig'
  rules <- many (parse_rule sig)
  eof
  return (sig, rules)

readFile :: FilePath -> IO (Signature, [(Term, Term)])
readFile f = do
  input <- Prelude.readFile f
  case parse parse_ari f input of
    Left err -> error (show err)
    Right r -> return r
