#ifndef PARALLEL_PROVERS_H
#define PARALLEL_PROVERS_H

#include "lemma_structures.h"

// --- VampireProver 实现 ---
class VampireProver : public Prover {
private:
    std::string executable_path;
    std::string lemma_generation_options; 

    //** 通用的fork/execv执行函数，通过输入的命令行参数，执行Vampire
    //** @param args 命令行参数列表
    //** @param output_file_path 输出文件路径
    //** @param time_limit_seconds 超时限制（秒）
    //** @return 执行结果：0表示成功，非0表示失败或超时
    int executeVampireProcess(const std::vector<std::string>& args,
                             const std::string& output_file_path,
                             int time_limit_seconds);

public:
    //** VampireProver构造函数
    //** @param exec_path Vampire可执行文件的路径
    //** @param lemma_gen_opts 引理生成选项，支持多种组合策略
    VampireProver(const std::string& exec_path, const std::string& lemma_gen_opts = "--show_new on");

    //** 获取推荐的引理生成选项(默认new)
    //** @param strategy 策略类型："new", "active", "passive"
    //** @return 对应的Vampire选项字符串
    static std::string getRecommendedLemmaOptions(const std::string& strategy = "new");

    //** 调用executeVampireProcess运行Vampire生成引理
    //** @param problem_path 问题文件路径
    //** @param output_log_path 输出日志文件路径  
    //** @param time_limit_seconds 时间限制（秒）
    //** @param memory_limit_mb 内存限制（MB），0表示不限制
    //** @return 命令是否成功执行（允许超时等非致命错误）
    bool runForLemmas(const std::string& problem_path,
                      const std::string& output_log_path,
                      int time_limit_seconds,
                      int memory_limit_mb = 0) override;

    //** 检查引理生成过程中是否已经找到证明
    //** 如果在引理生成阶段就找到了证明，则无需继续并行处理
    //** @param log_file_path 引理生成的日志文件路径
    //** @return 如果找到证明返回true，否则返回false
    bool checkProofFoundDuringLemmaGeneration(const std::string& log_file_path);

    //** 解析Vampire日志文件，提取引理
    //** 使用正则表达式匹配[SA] new/passive格式的行
    //** 提取子句编号、内容、来源信息、推导信息等详细信息
    //** @param log_file_path 日志文件路径
    //** @return 解析得到的引理列表，包含完整的来源信息和推导信息
    std::vector<Lemma> parseLogForLemmas(const std::string& log_file_path) override;

    //** 运行Vampire进行最终证明
    //** 使用fork/execv机制构建命令行并执行Vampire进行证明，将结果输出到文件
    //** 提供一致的PID控制和超时管理
    //** @param problem_path 问题文件路径
    //** @param result_file_path 结果输出文件路径
    //** @param time_limit_seconds 时间限制（秒）
    //** @param mode_options Vampire的模式选项，如"--mode casc"
    //** @param memory_limit_mb 内存限制（MB），0表示不限制
    //** @return 命令是否成功执行（允许超时等情况）
    bool runForProof(const std::string& problem_path,
                     const std::string& result_file_path,
                     int time_limit_seconds,
                     const std::string& mode_options,
                     int memory_limit_mb = 0) override;

    //** 检查Vampire证明结果
    //** 解析结果文件，查找TPTP标准的SZS状态信息
    //** @param result_file_path 结果文件路径
    //** @return SZS状态字符串，如"$SZS Status Theorem"、"$SZS Status Unknown"等
    std::string checkProofResult(const std::string& result_file_path) override;

    //** 获取证明器名称
    //** @return 返回"Vampire"字符串
    std::string getName() const override;

    //** 将TPTP文件转换为CNF格式
    //** 使用fork/execv机制和Vampire的CNF转换功能，将FOF/TFF格式转换为纯CNF
    //** 提供一致的PID控制和超时管理
    //** @param input_file_path 输入文件路径（可能包含FOF/TFF）
    //** @param output_cnf_path 输出CNF文件路径
    //** @param time_limit_seconds 转换时间限制（秒）
    //** @return 转换是否成功
    bool convertToCNF(const std::string& input_file_path,
                      const std::string& output_cnf_path,
                      int time_limit_seconds = 30);

    //** 清理CNF文件，移除非CNF内容
    //** 从Vampire输出中提取纯CNF子句，移除注释和状态信息
    //** @param raw_cnf_path 原始CNF文件路径
    //** @param clean_cnf_path 清理后的CNF文件路径
    //** @return CNF子句数量，失败时返回-1
    int cleanCNFFile(const std::string& raw_cnf_path,
                     const std::string& clean_cnf_path);

    //** 动态引理质量检查和生成
    //** 启动长时间的引理生成进程，定期检查最新引理质量，发现有效推导后及时停止
    //** @param cnf_file_path CNF文件路径
    //** @param log_file_path 日志文件路径
    //** @param base_time_limit 基础等待时间（秒）
    //** @param check_interval 检查间隔（秒）
    //** @param max_total_time 最大总时间（秒）
    //** @param memory_limit_mb 内存限制（MB）
    //** @return 是否成功生成有效引理
    bool runLemmasWithDynamicCheck(const std::string& cnf_file_path,
                                   const std::string& log_file_path,
                                   int base_time_limit,
                                   int check_interval,
                                   int max_total_time,
                                   int memory_limit_mb = 0);
};

// --- EProver 实现 ---
class EProver : public Prover {
private:
    std::string executable_path;

    //** 通用的fork/execv执行函数，执行E-Prover
    //** @param args 命令行参数列表
    //** @param output_file_path 输出文件路径
    //** @param time_limit_seconds 超时限制（秒）
    //** @return 执行结果：0表示成功，非0表示失败或超时
    int executeEProverProcess(const std::vector<std::string>& args,
                             const std::string& output_file_path,
                             int time_limit_seconds);

    //** 简化CNF语句，删除inference信息
    //** 将形如cnf(name, type, formula, inference(...), ['final']).的语句
    //** 简化为cnf(name, type, formula).格式，以便CSE能够正确读取
    //** @param cnf_clause 原始CNF子句字符串
    //** @return 简化后的CNF子句字符串
    std::string simplifyInferenceCNF(const std::string& cnf_clause);

public:
    //** EProver构造函数
    //** @param exec_path E-Prover可执行文件的路径
    EProver(const std::string& exec_path);

    //** E-Prover不支持引理生成，抛出异常
    //** 在混合策略中，引理生成由Vampire完成
    bool runForLemmas(const std::string& problem_path,
                      const std::string& output_log_path,
                      int time_limit_seconds,
                      int memory_limit_mb = 0) override;

    //** E-Prover不支持引理解析，返回空列表
    //** 在混合策略中，引理解析由Vampire完成
    std::vector<Lemma> parseLogForLemmas(const std::string& log_file_path) override;

    //** 运行E-Prover进行最终证明
    //** 使用E-Prover的自动策略调度和证明对象生成功能
    //** @param problem_path 问题文件路径
    //** @param result_file_path 结果输出文件路径
    //** @param time_limit_seconds 时间限制（秒）
    //** @param mode_options 模式选项（对E-Prover会被忽略，使用内置策略）
    //** @param memory_limit_mb 内存限制（MB），0表示不限制
    //** @return 命令是否成功执行
    bool runForProof(const std::string& problem_path,
                     const std::string& result_file_path,
                     int time_limit_seconds,
                     const std::string& mode_options,
                     int memory_limit_mb = 0) override;

    //** 检查E-Prover证明结果
    //** 解析E-Prover的输出，查找SZS状态信息
    //** E-Prover的输出格式与Vampire类似，但可能有细微差异
    //** @param result_file_path 结果文件路径
    //** @return SZS状态字符串
    std::string checkProofResult(const std::string& result_file_path) override;

    //** 获取证明器名称
    //** @return 返回"E-Prover"字符串
    std::string getName() const override;

    //** 将TPTP文件转换为CNF格式（使用E-Prover）
    //** 使用E-Prover的CNF转换功能，将FOF/TFF格式转换为纯CNF
    //** @param input_file_path 输入文件路径（可能包含FOF/TFF）
    //** @param output_cnf_path 输出CNF文件路径
    //** @param time_limit_seconds 转换时间限制（秒）
    //** @return 转换是否成功
    bool convertToCNF(const std::string& input_file_path,
                      const std::string& output_cnf_path,
                      int time_limit_seconds = 60);

    //** 清理CNF文件，移除非CNF内容（E-Prover版本）
    //** 从E-Prover输出中提取纯CNF子句
    //** @param raw_cnf_path 原始CNF文件路径
    //** @param clean_cnf_path 清理后的CNF文件路径
    //** @return CNF子句数量，失败时返回-1
    int cleanCNFFile(const std::string& raw_cnf_path,
                     const std::string& clean_cnf_path);
};

// --- CSEProver 实现 ---
class CSEProver : public Prover {
private:
    std::string executable_path;
    std::string work_directory;
    std::string cse_base_directory;
    std::string current_problem_name;

    //** 通用的fork/execv执行函数，执行CSE证明器
    //** @param args 命令行参数列表
    //** @param time_limit_seconds 超时限制（秒）
    //** @return 执行结果：0表示成功，非0表示失败或超时
    int executeCSEProcess(const std::vector<std::string>& args,
                         int time_limit_seconds);

public:
    //** CSEProver构造函数
    //** @param exec_path CSE可执行文件的路径
    //** @param work_dir 工作目录路径，默认为当前目录
    //** @param cse_base_dir CSE可执行文件的基目录
    CSEProver(const std::string& exec_path, const std::string& work_dir, const std::string& cse_base_dir);

    //** 设置当前问题名称（在运行引理生成前调用）
    //** @param problem_path 问题文件路径，将自动提取问题名称
    void setCurrentProblem(const std::string& problem_path);

    //** 运行CSE生成引理
    //** 使用CSE特有的命令行格式：/path/to/problem.p work/dir time
    //** @param problem_path 问题文件路径
    //** @param output_log_path 输出日志文件路径（暂时未使用，CSE有自己的输出机制）
    //** @param time_limit_seconds 时间限制（秒）
    //** @param memory_limit_mb 内存限制（MB），暂时未使用
    //** @return 命令是否成功执行
    bool runForLemmas(const std::string& problem_path,
                      const std::string& output_log_path,
                      int time_limit_seconds,
                      int memory_limit_mb = 0) override;

    //** CSE引理解析功能
    //** 解析CSE输出的单一.out文件
    //** @param log_file_path 日志文件路径（CSE不使用，而是使用内部的问题名称构建路径）
    //** @return 解析得到的引理列表
    std::vector<Lemma> parseLogForLemmas(const std::string& log_file_path) override;

    //** 运行CSE进行最终证明（暂未实现）
    //** @param problem_path 问题文件路径
    //** @param result_file_path 结果输出文件路径
    //** @param time_limit_seconds 时间限制（秒）
    //** @param mode_options 模式选项
    //** @param memory_limit_mb 内存限制（MB）
    //** @return 暂时返回false
    bool runForProof(const std::string& problem_path,
                     const std::string& result_file_path,
                     int time_limit_seconds,
                     const std::string& mode_options,
                     int memory_limit_mb = 0) override;

    //** 检查CSE证明结果（暂未实现）
    //** @param result_file_path 结果文件路径
    //** @return 暂时返回Unknown状态
    std::string checkProofResult(const std::string& result_file_path) override;

    //** 获取证明器名称
    //** @return 返回"CSE"字符串
    std::string getName() const override;

    //** 检查CSE引理生成过程中是否已经找到证明
    //** 通过检测CSE输出的.out文件中是否包含$false子句来判断是否找到证明
    //** 格式如：cnf(i_0_1057, plain, ($false), inference(...), ['proof']).
    //** @param cse_log_file CSE输出日志文件路径（用于提取问题名称构建.out文件路径）
    //** @return 如果在.out文件中找到$false子句返回true，否则返回false
    bool checkProofFoundDuringLemmaGeneration(const std::string& cse_log_file);

    //** 从问题文件路径中提取问题名称
    //** 例如：/path/to/GRP194+1.p -> GRP194+1
    //** @param problem_path 问题文件完整路径
    //** @return 问题名称（不含扩展名）
    std::string extractProblemName(const std::string& problem_path);

    //** 构建CSE输出目录路径
    //** 根据问题名称构建对应的输出文件夹路径
    //** 例如：GRP194+1 -> output/GRP194+1_short+eq_rstfp/
    //** @param problem_name 问题名称
    //** @return 输出目录路径
    std::string buildOutputDirectory(const std::string& problem_name);

    //** CSE动态引理检查机制
    //** 等待基础时间后，定期检查.out文件中的CNF子句行数
    //** 当检测到≥50行CNF子句时立即终止CSE进程
    //** @param problem_path 问题文件路径
    //** @param base_time_limit 基础等待时间（根据CNF子句数计算）
    //** @param check_interval 检查间隔（秒）
    //** @param max_total_time 最大总时间（秒）
    //** @param memory_limit_mb 内存限制（MB）
    //** @return 是否成功生成引理
    bool runLemmasWithDynamicCheck(const std::string& problem_path,
                                   int base_time_limit,
                                   int check_interval,
                                   int max_total_time,
                                   int memory_limit_mb = 0);
};

#endif // PARALLEL_PROVERS_H 