#include "parallel_provers.h"
#include "lemma_structures.h" // 包含file_exists等辅助函数

// --- VampireProver 实现 ---

VampireProver::VampireProver(const std::string& exec_path, const std::string& lemma_gen_opts)
    : executable_path(exec_path), lemma_generation_options(lemma_gen_opts) {}

std::string VampireProver::getRecommendedLemmaOptions(const std::string& strategy) {
    if (strategy == "new") {
        return "--show_new on";
    } else if (strategy == "active") {
        return "--show_active on";
    } else { 
        return "--show_passive on";
    }
}

int VampireProver::executeVampireProcess(const std::vector<std::string>& args,
                         const std::string& output_file_path,
                         int time_limit_seconds) {
    // 构建命令字符串用于日志
    std::string command_str;
    for (size_t i = 0; i < args.size(); ++i) {
        if (i > 0) command_str += " ";
        command_str += args[i];
    }
    command_str += " > \"" + output_file_path + "\" 2>&1";

    // 使用fork/exec启动进程
    pid_t child_pid = fork();
    if (child_pid == -1) {
        return -1;
    }
    // 子进程：重定向输出并执行vampire
    if (child_pid == 0) {
        // 在子进程中创建新的进程组，使其成为进程组的领导者
        if (setsid() == -1) {
        }
        
        // 重定向输出到文件
        int fd = open(output_file_path.c_str(), O_WRONLY | O_CREAT | O_TRUNC, 0644);
        if (fd != -1) {
            dup2(fd, STDOUT_FILENO);
            dup2(fd, STDERR_FILENO);
            close(fd);
        }
        
        // 准备execv参数
        std::vector<char*> argv_exec;
        for (const auto& arg : args) {
            argv_exec.push_back(const_cast<char*>(arg.c_str()));
        }
        argv_exec.push_back(nullptr);
        
        // 执行Vampire
        execv(executable_path.c_str(), argv_exec.data());
        
        // 如果execv失败
        exit(1);
    } else {
        // 父进程：等待子进程完成并处理超时
        int status;
        bool process_terminated = false;
        auto start_time = std::chrono::high_resolution_clock::now();
        
        while (!process_terminated) {
            // 检查是否超时
            auto current_time = std::chrono::high_resolution_clock::now();
            auto elapsed = std::chrono::duration_cast<std::chrono::seconds>(current_time - start_time);// 运行时间
            // 超时处理
            if (elapsed.count() >= time_limit_seconds) {
                
                // 向整个进程组发送SIGTERM信号
                kill(-child_pid, SIGTERM);
                
                // 等待一秒让进程退出
                for (int i = 0; i < 10; ++i) {
                    if (waitpid(child_pid, &status, WNOHANG) == child_pid) {
                        process_terminated = true;
                        break;
                    }
                    usleep(100000); 
                }
                
                // 如果还没退出，向整个进程组强制终止
                if (!process_terminated) {
                    kill(-child_pid, SIGKILL);
                    waitpid(child_pid, &status, 0);
                    process_terminated = true;
                }
                
                return 124; // 超时退出码
            }
            
            // 检查进程是否自然完成
            if (waitpid(child_pid, &status, WNOHANG) == child_pid) {
                process_terminated = true;
                return WEXITSTATUS(status);
            }
            
            // 短暂休眠后继续检查
            usleep(50000); // 50ms
        }
    }
    return -1; 
}

bool VampireProver::runForLemmas(const std::string& problem_path,
                  const std::string& output_log_path,
                  int time_limit_seconds,
                  int memory_limit_mb) {
    // 构建命令参数
    std::vector<std::string> args;
    args.push_back(executable_path);
    
    // 解析引理生成选项
    std::istringstream iss(lemma_generation_options);
    std::string token;
    while (iss >> token) {
        args.push_back(token);
    }
    
    args.push_back("-t");
    args.push_back(std::to_string(time_limit_seconds));
    
    // 添加内存限制参数（如果指定）
    if (memory_limit_mb > 0) {
        args.push_back("--memory_limit");
        args.push_back(std::to_string(memory_limit_mb));
    }
    
    args.push_back(problem_path);
    
    // 使用通用执行函数
    int result = executeVampireProcess(args, output_log_path, time_limit_seconds);
    
    // 检查执行结果
    if (result == 0) {
        return true;
    } else if (result == 124) {
        return true; 
    } else {
        return false;
    }
}

bool VampireProver::checkProofFoundDuringLemmaGeneration(const std::string& log_file_path) {
    std::ifstream log_file(log_file_path);
    std::string line;
    
    if (!log_file.is_open()) {
        return false;
    }
    
    // 查找证明相关的状态信息
    std::regex refutation_regex(R"(Termination reason:\s*Refutation(?!\s+not\s+found))");
    std::regex szs_unsatisfiable_regex(R"(SZS\s+status\s+Unsatisfiable)");
    std::regex szs_theorem_regex(R"(SZS\s+status\s+Theorem)");
    std::regex szs_status_success_regex(R"(^\s*%\s+SZS\s+status\s+(Theorem|Unsatisfiable)\s+for)");
    
    while (std::getline(log_file, line)) {
        if (std::regex_search(line, refutation_regex)) {
            log_file.close();
            return true;
        }
        if (std::regex_search(line, szs_status_success_regex)) {
            log_file.close();
            return true;
        }
        if (std::regex_search(line, szs_unsatisfiable_regex) ||
            std::regex_search(line, szs_theorem_regex)) {
            log_file.close();
            return true;
        }
    }
    
    log_file.close();
    return false;
}

std::vector<Lemma> VampireProver::parseLogForLemmas(const std::string& log_file_path) {
    std::vector<Lemma> lemmas;
    std::ifstream log_file(log_file_path);
    std::string line;
    int lemma_counter = 1;
    int processed_lines = 0;
    int matched_lines = 0;

    if (!log_file.is_open()) {
        return lemmas;
    }

    // 预分配内存以避免频繁重新分配
    lemmas.reserve(30000); 

    // 恢复复杂但准确的正则表达式，用于正确解析推导信息
    std::regex lemma_line_regex;
    try {
        // 匹配格式：[SA] new/active/passive: 数字. 内容 [推导信息]
        lemma_line_regex = std::regex(R"(^\s*\[SA\]\s+(new|active|passive):\s*(\d+)\.\s*(.*?)(?:\s+\[([^\]]*)\])?\s*$)");
    } catch (const std::exception& e) {
        return lemmas;
    }

    while (std::getline(log_file, line)) {
        processed_lines++;
        
        // 检查提前停止标志
        if (proof_found.load()) {
            break;
        }
        
        // 基本长度检查，跳过过长的行
        if (line.length() > 2000) {
            continue;
        }
        
        try {
            std::smatch match;
            if (std::regex_search(line, match, lemma_line_regex)) {
                matched_lines++;
                
                if (match.size() >= 4) {
                    std::string derivation_type = match[1].matched ? match[1].str() : "";
                    std::string clause_num_str = match[2].matched ? match[2].str() : "";
                    std::string content = match[3].matched ? match[3].str() : "";
                    std::string inference_info = (match.size() > 4 && match[4].matched) ? match[4].str() : "";

                    if (derivation_type.empty() || clause_num_str.empty() || content.empty()) {
                        continue;
                    }

                    // 清理内容
                    content.erase(0, content.find_first_not_of(" \t\n\r\f\v"));
                    content.erase(content.find_last_not_of(" \t\n\r\f\v") + 1);
                    
                    // 限制内容长度
                    if (content.length() > 800) {
                        content = content.substr(0, 800);
                    }
                    
                    // 清理问题字符
                    if (content.find("\"") != std::string::npos) {
                        std::replace(content.begin(), content.end(), '\"', '\'');
                    }

                    // 解析子句编号
                    int clause_number = -1;
                    try {
                        if (!clause_num_str.empty() && clause_num_str.length() <= 10) {
                            clause_number = std::stoi(clause_num_str);
                        }
                    } catch (const std::exception& e) {
                        continue;
                    }

                    // 区分input和推导引理
                    std::string inference_rule;
                    std::vector<int> parent_clause_numbers;
                    std::string source_info = inference_info; // 保存原始推导信息
                    
                    // 判断是否为input子句并设置source_info
                    if (!inference_info.empty()) {
                        if (inference_info.find("input(") != std::string::npos || 
                            inference_info.find("definition unfolding") != std::string::npos) {
                            // input(axiom) 和 definition unfolding 都属于输入阶段
                            source_info = "input";
                        } else {
                            // 解析推导信息，提取推理规则和父子句
                            try {
                                // 查找常见的推理规则
                                if (inference_info.find("resolution") != std::string::npos) {
                                    inference_rule = "resolution";
                                } else if (inference_info.find("superposition") != std::string::npos) {
                                    inference_rule = "superposition";
                                } else if (inference_info.find("definition unfolding") != std::string::npos) {
                                    inference_rule = "definition_unfolding";
                                } else if (inference_info.find("forward demodulation") != std::string::npos) {
                                    inference_rule = "forward_demodulation";
                                } else if (inference_info.find("backward demodulation") != std::string::npos) {
                                    inference_rule = "backward_demodulation";
                                } else if (inference_info.find("factoring") != std::string::npos) {
                                    inference_rule = "factoring";
                                } else {
                                    inference_rule = "other";
                                }
                                
                                // 尝试提取数字作为父子句（简化处理）
                                std::regex number_regex(R"(\b(\d+)\b)");
                                std::sregex_iterator numbers_begin(inference_info.begin(), inference_info.end(), number_regex);
                                std::sregex_iterator numbers_end;
                                
                                for (std::sregex_iterator i = numbers_begin; i != numbers_end && parent_clause_numbers.size() < 5; ++i) {
                                    try {
                                        int parent_num = std::stoi((*i)[1].str());
                                        if (parent_num > 0 && parent_num != clause_number) { // 避免自引用
                                            parent_clause_numbers.push_back(parent_num);
                                        }
                                    } catch (...) {
                                        continue;
                                    }
                                }
                            } catch (const std::exception& e) {
                                // 忽略推导信息解析错误，但保留原始信息
                            }
                        }
                    } else {
                        // 没有推导信息的引理，可能是早期的input子句
                        source_info = "input";
                    }

                    // 基本过滤：排除明显无用的子句
                    if (!content.empty() && 
                        content.find("$false") == std::string::npos &&
                        content.find("$true") == std::string::npos &&
                        content.length() >= 3 && content.length() <= 800) {
                        
                        try {
                            Lemma l;
                            l.id = "vampireLemma" + std::to_string(lemma_counter++);
                            l.role = "axiom";
                            l.content = std::move(content);
                            l.derivation_type = std::move(derivation_type);
                            l.clause_number = clause_number;
                            l.source_info = std::move(source_info); // 正确设置source_info
                            l.avatar_id = ""; // 暂时不处理avatar
                            l.inference_rule = std::move(inference_rule);
                            l.parent_clause_numbers = std::move(parent_clause_numbers);
                            l.derivation_depth = -1; // 稍后计算
                            
                            lemmas.push_back(std::move(l));
                        } catch (const std::exception& e) {
                            continue;
                        }
                    }
                }
            }
        } catch (const std::exception& e) {
            continue;
        }
    }
    
    log_file.close();
    
    // 统计input和非input引理
    int input_count = 0;
    int derived_count = 0;
    for (const auto& lemma : lemmas) {
        if (lemma.source_info == "input") {
            input_count++;
        } else {
            derived_count++;
        }
    }
    
    return lemmas;
}

bool VampireProver::runForProof(const std::string& problem_path,
                 const std::string& result_file_path,
                 int time_limit_seconds,
                 const std::string& mode_options,
                 int memory_limit_mb) {
    // 构建命令参数
    std::vector<std::string> args;
    args.push_back(executable_path);
    
    // 解析模式选项
    std::istringstream iss(mode_options);
    std::string token;
    while (iss >> token) {
        args.push_back(token);
    }
    
    args.push_back("-t");
    args.push_back(std::to_string(time_limit_seconds));
    
    // 添加内存限制参数（如果指定）
    if (memory_limit_mb > 0) {
        args.push_back("--memory_limit");
        args.push_back(std::to_string(memory_limit_mb));
    }
    
    args.push_back(problem_path);
    
    // 使用通用执行函数
    int result = executeVampireProcess(args, result_file_path, time_limit_seconds);
    
    // 检查执行结果
    if (result == 0) {
        return true;
    } else if (result == 124) {
        return true; // 超时也被认为是可接受的
    } else {
        return false;
    }
}

std::string VampireProver::checkProofResult(const std::string& result_file_path) {
    std::ifstream result_file(result_file_path);
    std::string line;
    std::string szs_status = "$SZS Status Unknown"; // 默认状态

    if (!result_file.is_open()) {
        return "$SZS Status Error: Result file not found or accessible.";
    }

    std::regex szs_status_regex1(R"(^\s*%\s*SZS\s+status\s+(\S+)\s+for)"); 
    std::regex szs_status_regex2(R"(^\s*\$SZS\s+Status\s+(\S+)\s*$)"); 
    std::regex szs_status_regex3(R"(^\s*SZS\s+Status\s+(\S+)\s*$)"); 

    while (std::getline(result_file, line)) {
        std::smatch match;
        
        // 尝试匹配第一种格式
        if (std::regex_search(line, match, szs_status_regex1)) {
            if (match.size() > 1) {
                szs_status = "$SZS Status " + match[1].str();
                break;
            }
        }
        // 尝试匹配第二种格式
        else if (std::regex_search(line, match, szs_status_regex2)) {
            if (match.size() > 1) {
                szs_status = "$SZS Status " + match[1].str();
                break;
            }
        }
        // 尝试匹配第三种格式
        else if (std::regex_search(line, match, szs_status_regex3)) {
            if (match.size() > 1) {
                szs_status = "$SZS Status " + match[1].str();
                break;
            }
        }
    }
    result_file.close();
    return szs_status;
}

std::string VampireProver::getName() const { 
    return "Vampire"; 
}

bool VampireProver::convertToCNF(const std::string& input_file_path,
                  const std::string& output_cnf_path,
                  int time_limit_seconds) {
    // 构建命令参数
    std::vector<std::string> args;
    args.push_back(executable_path);
    args.push_back("--mode");
    args.push_back("clausify");
    args.push_back("-t");
    args.push_back(std::to_string(time_limit_seconds));
    args.push_back(input_file_path);
    
    // 使用通用执行函数
    int result = executeVampireProcess(args, output_cnf_path, time_limit_seconds);
    
    // 检查执行结果
    if (result != 0 && result != 124) {
        return false;
    }
    
    // 检查输出文件是否生成且非空
    std::ifstream check_file(output_cnf_path);
    if (!check_file.is_open()) {
        return false;
    }

    // 检查文件是否有内容
    std::string line;
    bool has_cnf_content = false;
    while (std::getline(check_file, line)) {
        if (line.find("cnf(") != std::string::npos) {
            has_cnf_content = true;
            break;
        }
    }
    check_file.close();
    
    if (!has_cnf_content) {
        return false;
    }

    return true;
}

int VampireProver::cleanCNFFile(const std::string& raw_cnf_path,
                 const std::string& clean_cnf_path) {
    std::ifstream input_file(raw_cnf_path);
    std::ofstream output_file(clean_cnf_path);
    
    if (!input_file.is_open() || !output_file.is_open()) {
        return -1;
    }

    std::string line;
    int cnf_count = 0;
    bool in_cnf_clause = false;
    std::string current_cnf;
    
    // 添加文件头注释
    output_file << "% CNF file generated by Vampire\n";
    output_file << "% Cleaned and processed for parallel proving\n\n";
    
    while (std::getline(input_file, line)) {
        // 检查是否是CNF子句的开始
        if (line.find("cnf(") != std::string::npos) {
            in_cnf_clause = true;
            current_cnf = line;
            
            // 检查是否是单行CNF子句（以).结尾）
            if (line.find(").") != std::string::npos) {
                output_file << current_cnf << "\n";
                cnf_count++;
                in_cnf_clause = false;
                current_cnf.clear();
            }
        }
        // 如果正在处理多行CNF子句
        else if (in_cnf_clause) {
            current_cnf += "\n" + line;
            
            // 检查是否是CNF子句的结束（以).结尾）
            if (line.find(").") != std::string::npos) {
                output_file << current_cnf << "\n\n";
                cnf_count++;
                in_cnf_clause = false;
                current_cnf.clear();
            }
        }
        else if (line.find("% SZS") != std::string::npos) {
            output_file << line << "\n";
        }
    }
    
    input_file.close();
    output_file.close();
    
    return cnf_count;
}

// --- EProver 实现 ---

EProver::EProver(const std::string& exec_path) : executable_path(exec_path) {}

int EProver::executeEProverProcess(const std::vector<std::string>& args,
                     const std::string& output_file_path,
                     int time_limit_seconds) {
    // 构建命令字符串用于日志
    std::string command_str;
    for (size_t i = 0; i < args.size(); ++i) {
        if (i > 0) command_str += " ";
        command_str += args[i];
    }
    command_str += " > \"" + output_file_path + "\" 2>&1";

    // 使用fork/exec启动进程
    pid_t child_pid = fork();
    if (child_pid == -1) {
        return -1;
    }
    
    // 子进程：重定向输出并执行E-Prover
    if (child_pid == 0) {
        // 在子进程中创建新的进程组，使其成为进程组的领导者
        if (setsid() == -1) {
        }
        
        // 重定向输出到文件
        int fd = open(output_file_path.c_str(), O_WRONLY | O_CREAT | O_TRUNC, 0644);
        if (fd != -1) {
            dup2(fd, STDOUT_FILENO);
            dup2(fd, STDERR_FILENO);
            close(fd);
        }
        
        // 准备execv参数
        std::vector<char*> argv_exec;
        for (const auto& arg : args) {
            argv_exec.push_back(const_cast<char*>(arg.c_str()));
        }
        argv_exec.push_back(nullptr);
        
        // 执行E-Prover
        execv(executable_path.c_str(), argv_exec.data());
        
        // 如果execv失败
        exit(1);
    } else {
        // 父进程：等待子进程完成并处理超时
        int status;
        bool process_terminated = false;
        auto start_time = std::chrono::high_resolution_clock::now();
        
        while (!process_terminated) {
            // 检查是否超时
            auto current_time = std::chrono::high_resolution_clock::now();
            auto elapsed = std::chrono::duration_cast<std::chrono::seconds>(current_time - start_time);
            
            // 超时处理
            if (elapsed.count() >= time_limit_seconds) {
                
                // 向整个进程组发送SIGTERM信号
                kill(-child_pid, SIGTERM);
                
                // 等待一秒让进程退出
                for (int i = 0; i < 10; ++i) {
                    if (waitpid(child_pid, &status, WNOHANG) == child_pid) {
                        process_terminated = true;
                        break;
                    }
                    usleep(100000); 
                }
                
                // 如果还没退出，向整个进程组强制终止
                if (!process_terminated) {
                    kill(-child_pid, SIGKILL);
                    waitpid(child_pid, &status, 0);
                    process_terminated = true;
                }
                
                return 124; // 超时退出码
            }
            
            // 检查进程是否自然完成
            if (waitpid(child_pid, &status, WNOHANG) == child_pid) {
                process_terminated = true;
                return WEXITSTATUS(status);
            }
            
            // 短暂休眠后继续检查
            usleep(50000); // 50ms
        }
    }
    return -1; 
}

bool EProver::runForLemmas(const std::string& problem_path,
                  const std::string& output_log_path,
                  int time_limit_seconds,
                  int memory_limit_mb) {
    (void)problem_path; // 消除未使用参数警告
    (void)output_log_path;
    (void)time_limit_seconds;
    (void)memory_limit_mb;
    
    return false;
}

std::vector<Lemma> EProver::parseLogForLemmas(const std::string& log_file_path) {
    (void)log_file_path; // 消除未使用参数警告
    
    return std::vector<Lemma>();
}

bool EProver::runForProof(const std::string& problem_path,
                 const std::string& result_file_path,
                 int time_limit_seconds,
                 const std::string& mode_options,
                 int memory_limit_mb) {
    // 构建E-Prover命令参数
    std::vector<std::string> args;
    args.push_back(executable_path);
    
    // 如果提供了mode_options，解析并使用它们
    if (!mode_options.empty()) {
        std::istringstream iss(mode_options);
        std::string token;
        while (iss >> token) {
            args.push_back(token);
        }
    } else {
        // 默认的E-Prover 2.6单进程模式（作为后备）
        args.push_back("--auto");               // 单进程自动策略
        args.push_back("--proof-object");       // 生成证明对象
        args.push_back("--output-level=2");     // 输出级别2
    }
    
    args.push_back("--cpu-limit=" + std::to_string(time_limit_seconds));
    
    // 添加内存限制（如果指定）
    if (memory_limit_mb > 0) {
        args.push_back("--memory-limit=" + std::to_string(memory_limit_mb));
    }
    
    args.push_back(problem_path);
    
    // 使用通用执行函数
    int result = executeEProverProcess(args, result_file_path, time_limit_seconds);
    
    // 检查执行结果
    if (result == 0) {
        return true;
    } else if (result == 124) {
        return true; // 超时也被认为是可接受的
    } else {
        return false;
    }
}

std::string EProver::checkProofResult(const std::string& result_file_path) {
    std::ifstream result_file(result_file_path);
    std::string line;
    std::string szs_status = "$SZS Status Unknown"; // 默认状态

    if (!result_file.is_open()) {
        return "$SZS Status Error: Result file not found or accessible.";
    }

    // E-Prover的SZS状态输出格式
    std::regex szs_status_regex1(R"(^\s*#\s*SZS\s+status\s+(\S+)\s+for)");  // E-Prover格式
    std::regex szs_status_regex2(R"(^\s*%\s*SZS\s+status\s+(\S+)\s+for)");  // 标准格式
    std::regex szs_status_regex3(R"(^\s*SZS\s+Status\s+(\S+)\s*$)");        // 简化格式
    std::regex szs_status_regex4(R"(^\s*#\s*Proof\s+found!)");              // E-Prover特有的证明找到标志
    std::regex szs_status_regex5(R"(^\s*#\s*No\s+proof\s+found)");          // E-Prover特有的未找到证明标志

    while (std::getline(result_file, line)) {
        std::smatch match;
        
        // 尝试匹配E-Prover格式
        if (std::regex_search(line, match, szs_status_regex1)) {
            if (match.size() > 1) {
                szs_status = "$SZS Status " + match[1].str();
                break;
            }
        }
        // 尝试匹配标准格式
        else if (std::regex_search(line, match, szs_status_regex2)) {
            if (match.size() > 1) {
                szs_status = "$SZS Status " + match[1].str();
                break;
            }
        }
        // 尝试匹配简化格式
        else if (std::regex_search(line, match, szs_status_regex3)) {
            if (match.size() > 1) {
                szs_status = "$SZS Status " + match[1].str();
                break;
            }
        }
        // E-Prover特有的证明找到标志
        else if (std::regex_search(line, szs_status_regex4)) {
            szs_status = "$SZS Status Theorem";
            break;
        }
        // E-Prover特有的未找到证明标志
        else if (std::regex_search(line, szs_status_regex5)) {
            szs_status = "$SZS Status Unknown";
            // 不break，继续寻找更具体的状态
        }
    }
    result_file.close();
    return szs_status;
}

std::string EProver::getName() const {
    return "E-Prover";
}

bool EProver::convertToCNF(const std::string& input_file_path,
                          const std::string& output_cnf_path,
                          int time_limit_seconds) {
    // 构建E-Prover CNF转换命令参数
    std::vector<std::string> args;
    args.push_back(executable_path);
    args.push_back("--cnf");                    // CNF转换模式
    args.push_back("--cpu-limit=" + std::to_string(time_limit_seconds));
    args.push_back("-p");                       // 添加CNF语句的来源信息
    args.push_back(input_file_path);
    
    // 使用通用执行函数
    int result = executeEProverProcess(args, output_cnf_path, time_limit_seconds);
    
    // 检查执行结果
    if (result != 0 && result != 124) {
        return false;
    }
    
    // 检查输出文件是否生成且非空
    std::ifstream check_file(output_cnf_path);
    if (!check_file.is_open()) {
        return false;
    }

    // 检查文件是否有内容
    std::string line;
    bool has_cnf_content = false;
    while (std::getline(check_file, line)) {
        if (line.find("cnf(") != std::string::npos) {
            has_cnf_content = true;
            break;
        }
    }
    check_file.close();
    
    if (!has_cnf_content) {
        return false;
    }

    return true;
}

std::string EProver::simplifyInferenceCNF(const std::string& cnf_clause) {
    // 查找cnf(开头
    size_t cnf_start = cnf_clause.find("cnf(");
    if (cnf_start == std::string::npos) {
        return cnf_clause; // 如果不是CNF语句，直接返回原始内容
    }
    
    // 从cnf(后开始查找逗号
    size_t pos = cnf_start + 4; // 跳过"cnf("
    int comma_count = 0;
    int paren_depth = 0;
    bool in_string = false;
    
    for (size_t i = pos; i < cnf_clause.length(); ++i) {
        char c = cnf_clause[i];
        
        // 处理字符串中的引号
        if (c == '"' || c == '\'') {
            in_string = !in_string;
            continue;
        }
        
        // 如果在字符串中，跳过所有字符
        if (in_string) {
            continue;
        }
        
        // 处理括号深度
        if (c == '(') {
            paren_depth++;
        } else if (c == ')') {
            paren_depth--;
        } else if (c == ',' && paren_depth == 0) {
            // 只有在括号深度为0时的逗号才算数
            comma_count++;
            if (comma_count == 3) {
                // 找到第三个逗号，截取到这里
                std::string simplified = cnf_clause.substr(0, i) + ").";
                return simplified;
            }
        }
    }
    
    // 如果没有找到第三个逗号，可能是格式不标准，返回原始内容
    return cnf_clause;
}

int EProver::cleanCNFFile(const std::string& raw_cnf_path,
                         const std::string& clean_cnf_path) {
    std::ifstream input_file(raw_cnf_path);
    std::ofstream output_file(clean_cnf_path);
    
    // 创建完整信息文件（用于后续证明输出）
    std::string full_info_path = clean_cnf_path;
    size_t dot_pos = full_info_path.find_last_of(".");
    if (dot_pos != std::string::npos) {
        full_info_path.insert(dot_pos, "_full");
    } else {
        full_info_path += "_full";
    }
    std::ofstream full_info_file(full_info_path);
    
    if (!input_file.is_open() || !output_file.is_open() || !full_info_file.is_open()) {
        return -1;
    }

    std::string line;
    int cnf_count = 0;
    bool in_cnf_clause = false;
    bool in_fof_clause = false;
    std::string current_cnf;
    std::string current_fof;
    
    while (std::getline(input_file, line)) {
        // 跳过注释行（以 # 开头的行），但SZS状态信息除外
        if (line.find("#") == 0 && line.find("# SZS") == std::string::npos) {
            continue; // 不写入_full.cnf文件
        }
        
        // 检查是否是FOF语句的开始
        if (line.find("fof(") != std::string::npos) {
            in_fof_clause = true;
            current_fof = line;
            
            // 检查是否是单行FOF语句（以).结尾）
            if (line.find(").") != std::string::npos) {
                // _full文件：保存所有FOF语句
                full_info_file << current_fof << "\n";
                in_fof_clause = false;
                current_fof.clear();
            }
        }
        // 检查是否是CNF子句的开始
        else if (line.find("cnf(") != std::string::npos) {
            in_cnf_clause = true;
            current_cnf = line;
            
            // 检查是否是单行CNF子句（以).结尾）
            if (line.find(").") != std::string::npos) {
                // _full文件：保存所有CNF子句的完整信息
                full_info_file << current_cnf << "\n";
                
                // 普通文件：只保留包含['final']标记的CNF语句，并简化
                if (current_cnf.find("['final']") != std::string::npos) {
                    std::string simplified_cnf = simplifyInferenceCNF(current_cnf);
                    output_file << simplified_cnf << "\n";
                    cnf_count++;
                }
                in_cnf_clause = false;
                current_cnf.clear();
            }
        }
        // 如果正在处理多行FOF语句
        else if (in_fof_clause) {
            current_fof += "\n" + line;
            
            // 检查是否是FOF语句的结束（以).结尾）
            if (line.find(").") != std::string::npos) {
                // _full文件：保存所有FOF语句
                full_info_file << current_fof << "\n\n";
                in_fof_clause = false;
                current_fof.clear();
            }
        }
        // 如果正在处理多行CNF子句
        else if (in_cnf_clause) {
            current_cnf += "\n" + line;
            
            // 检查是否是CNF子句的结束（以).结尾）
            if (line.find(").") != std::string::npos) {
                // _full文件：保存所有CNF子句的完整信息
                full_info_file << current_cnf << "\n\n";
                
                // 普通文件：只保留包含['final']标记的CNF语句，并简化
                if (current_cnf.find("['final']") != std::string::npos) {
                    std::string simplified_cnf = simplifyInferenceCNF(current_cnf);
                    output_file << simplified_cnf << "\n\n";
                    cnf_count++;
                }
                in_cnf_clause = false;
                current_cnf.clear();
            }
        }
        // 保留SZS状态信息
        else if (line.find("% SZS") != std::string::npos || 
                 line.find("# SZS") != std::string::npos) {
            output_file << line << "\n";
            full_info_file << line << "\n";
        }
    }
    
    input_file.close();
    output_file.close();
    full_info_file.close();
    
    return cnf_count;
}

bool VampireProver::runLemmasWithDynamicCheck(const std::string& cnf_file_path,
                                   const std::string& log_file_path,
                                   int base_time_limit,
                                   int check_interval,
                                   int max_total_time,
                                   int memory_limit_mb) {
    
    // 构建命令参数
    std::vector<std::string> args;
    args.push_back(executable_path);
    
    // 解析引理生成选项
    std::istringstream iss(lemma_generation_options);
    std::string token;
    while (iss >> token) {
        args.push_back(token);
    }
    
    args.push_back("-t");
    args.push_back(std::to_string(max_total_time)); // 使用最大时间作为vampire的时间限制
    
    if (memory_limit_mb > 0) {
        args.push_back("--memory_limit");
        args.push_back(std::to_string(memory_limit_mb));
    }
    
    args.push_back(cnf_file_path);
    
    // 构建命令字符串
    std::string command_str = executable_path;
    for (size_t i = 1; i < args.size(); ++i) {
        command_str += " " + args[i];
    }
    command_str += " > \"" + log_file_path + "\" 2>&1";
    
    // 使用fork启动vampire进程
    pid_t vampire_pid = fork();
    if (vampire_pid == -1) {
        return false;
    }
    
    if (vampire_pid == 0) {
        // 子进程：重定向输出并执行vampire
        int fd = open(log_file_path.c_str(), O_WRONLY | O_CREAT | O_TRUNC, 0644);
        if (fd != -1) {
            dup2(fd, STDOUT_FILENO);
            dup2(fd, STDERR_FILENO);
            close(fd);
        }
        
        // 准备execv参数
        std::vector<char*> argv_exec;
        for (const auto& arg : args) {
            argv_exec.push_back(const_cast<char*>(arg.c_str()));
        }
        argv_exec.push_back(nullptr);
        
        execv(executable_path.c_str(), argv_exec.data());
        
        // 如果execv失败
        exit(1);
    } else {
        // 父进程：定期检查引理质量
        auto start_time = std::chrono::high_resolution_clock::now();
        bool found_derived_lemmas = false;
        bool process_terminated = false;
        
        // 等待基础时间，但期间检查提前停止标志
        for (int i = 0; i < base_time_limit; ++i) {
            if (proof_found.load()) {
                found_derived_lemmas = true;
                break;
            }
            sleep(1); // 每秒检查一次
        }
        
        // 开始定期检查
        while (!found_derived_lemmas && !process_terminated) {
            auto current_time = std::chrono::high_resolution_clock::now();
            auto elapsed = std::chrono::duration_cast<std::chrono::seconds>(current_time - start_time);
            
            if (proof_found.load()) {
                break;
            }
            
            // 检查总时间是否超限
            if (elapsed.count() >= max_total_time) {
                break;
            }
            
            // 检查进程是否自然结束
            int status;
            if (waitpid(vampire_pid, &status, WNOHANG) == vampire_pid) {
                process_terminated = true;
                break;
            }
            
            //只解析文件末尾的新引理，而不是整个文件
            static std::streampos last_file_position = 0;
            static int last_total_count = 0;
            
            std::ifstream log_file(log_file_path);
            if (!log_file.is_open()) {
                sleep(check_interval);
                continue;
            }
            
            // 跳到上次读取的位置
            log_file.seekg(last_file_position);
            
            // 解析新增的引理
            std::vector<Lemma> new_lemmas;
            std::string line;
            std::streampos current_position = last_file_position;
            
            while (std::getline(log_file, line)) {
                // 记录每行读取前的位置
                current_position = log_file.tellg();
                
                if (line.find("new:") != std::string::npos || line.find("passive:") != std::string::npos) {
                    // 只检查关键信息
                    Lemma lemma;
                    
                    // 解析子句编号 
                    size_t clause_num_pos = line.find(" (");
                    if (clause_num_pos != std::string::npos) {
                        size_t end_pos = line.find(")", clause_num_pos);
                        if (end_pos != std::string::npos) {
                            std::string clause_num_str = line.substr(clause_num_pos + 2, end_pos - clause_num_pos - 2);
                            try {
                                lemma.clause_number = std::stoi(clause_num_str);
                            } catch (...) {
                                lemma.clause_number = -1;
                            }
                        }
                    }
                    
                    // 更精确的来源信息检查
                    if (line.find("[input") != std::string::npos || 
                        line.find("[definition unfolding") != std::string::npos) {
                        lemma.source_info = "input";
                    } else {
                        lemma.source_info = "";  // 非input，可能是推导引理
                    }
                    
                    // 更全面的推理规则检查（表明是推导引理）
                    if (line.find("[resolution") != std::string::npos || 
                        line.find("[superposition") != std::string::npos ||
                        line.find("[forward demodulation") != std::string::npos ||
                        line.find("[backward demodulation") != std::string::npos ||
                        line.find("[factoring") != std::string::npos ||
                        line.find("[subsumption resolution") != std::string::npos ||
                        (line.find("[") != std::string::npos && 
                         line.find("input") == std::string::npos && 
                         line.find("definition unfolding") == std::string::npos)) {
                        lemma.parent_clause_numbers.push_back(1); // 非空表示有父子句
                    }
                    
                    new_lemmas.push_back(lemma);
                }
            }
            
            if (current_position != static_cast<std::streampos>(-1)) {
                last_file_position = current_position;
            } else if (!new_lemmas.empty()) {
                // 如果读取到了内容但位置无效，保持上次的有效位置不变
            }
            
            log_file.close();
            
            // 计算总引理数（累加）
            int current_total = last_total_count + new_lemmas.size();
            last_total_count = current_total;
            
            if (!new_lemmas.empty() || current_total > 0) {
                // 检查最近的引理质量
                const int CHECK_COUNT = 500;
                int check_count = std::min(CHECK_COUNT, static_cast<int>(new_lemmas.size()));
                
                int recent_derived_count = 0;
                int start_check = std::max(0, static_cast<int>(new_lemmas.size()) - check_count);
                for (int i = start_check; i < static_cast<int>(new_lemmas.size()); ++i) {
                    const auto& lemma = new_lemmas[i];
                    if (lemma.source_info.find("input") == std::string::npos && 
                        !lemma.parent_clause_numbers.empty()) {
                        recent_derived_count++;
                    }
                }
                
                // 统计input引理数量
                int input_count = 0;
                for (int i = start_check; i < static_cast<int>(new_lemmas.size()); ++i) {
                    if (new_lemmas[i].source_info == "input") {
                        input_count++;
                    }
                }
                
                // 如果发现推导引理，准备停止
                if (recent_derived_count > 0) {
                    found_derived_lemmas = true;
                    break;
                }
            }
            
            // 等待下一次检查
            sleep(check_interval);
        }
        
        // 如果发现了推导引理或达到时间限制，停止vampire进程
        if (!process_terminated) {
            kill(vampire_pid, SIGTERM);
            
            // 给进程一些时间退出
            for (int i = 0; i < 10; ++i) {
                int status;
                if (waitpid(vampire_pid, &status, WNOHANG) == vampire_pid) {
                    process_terminated = true;
                    break;
                }
                usleep(100000); // 100ms
            }
            
            // 如果仍未退出，强制终止
            if (!process_terminated) {
                kill(vampire_pid, SIGKILL);
                waitpid(vampire_pid, nullptr, 0);
            }
        }
        
        auto end_time = std::chrono::high_resolution_clock::now();
        auto total_duration = std::chrono::duration_cast<std::chrono::seconds>(end_time - start_time);
        
        if (found_derived_lemmas) {
            return true;
        } else {
            return false; // 仍然返回false，但不影响后续处理
        }
    }
    
    return false;
}

// --- CSEProver 实现 ---

CSEProver::CSEProver(const std::string& exec_path, const std::string& work_dir, const std::string& cse_base_dir)
    : executable_path(exec_path), work_directory(work_dir), cse_base_directory(cse_base_dir), current_problem_name("unknown") {}

void CSEProver::setCurrentProblem(const std::string& problem_path) {
    current_problem_name = extractProblemName(problem_path);
}

int CSEProver::executeCSEProcess(const std::vector<std::string>& args,
                                int time_limit_seconds) {
    // 构建命令字符串用于日志
    std::string command_str;
    for (size_t i = 0; i < args.size(); ++i) {
        if (i > 0) command_str += " ";
        command_str += args[i];
    }

    // 使用fork/exec启动进程
    pid_t child_pid = fork();
    if (child_pid == -1) {
        return -1;
    }
    
    // 子进程：执行CSE
    if (child_pid == 0) {
        // 在子进程中创建新的进程组，使其成为进程组的领导者
        if (setsid() == -1) {
        }
        
        // 重定向CSE的输出到/dev/null，避免在控制台显示
        int dev_null = open("/dev/null", O_WRONLY);
        if (dev_null != -1) {
            dup2(dev_null, STDOUT_FILENO);
            dup2(dev_null, STDERR_FILENO);
            close(dev_null);
        }
        
        // 准备execv参数
        std::vector<char*> argv_exec;
        for (const auto& arg : args) {
            argv_exec.push_back(const_cast<char*>(arg.c_str()));
        }
        argv_exec.push_back(nullptr);
        
        // 执行CSE
        execv(executable_path.c_str(), argv_exec.data());
        
        // 如果execv失败
        exit(1);
    } else {
        // 父进程：等待子进程完成并处理超时
        int status;
        bool process_terminated = false;
        auto start_time = std::chrono::high_resolution_clock::now();
        
        while (!process_terminated) {
            // 检查提前停止标志
            if (proof_found.load()) {
                kill(-child_pid, SIGTERM);
                // 等待进程退出
                for (int i = 0; i < 10; ++i) {
                    if (waitpid(child_pid, &status, WNOHANG) == child_pid) {
                        process_terminated = true;
                        break;
                    }
                    usleep(100000); 
                }
                if (!process_terminated) {
                    kill(-child_pid, SIGKILL);
                    waitpid(child_pid, &status, 0);
                }
                return 130; // 提前停止退出码
            }
            
            // 检查是否超时
            auto current_time = std::chrono::high_resolution_clock::now();
            auto elapsed = std::chrono::duration_cast<std::chrono::seconds>(current_time - start_time);
            
            if (elapsed.count() >= time_limit_seconds) {
                // 向整个进程组发送SIGTERM信号
                kill(-child_pid, SIGTERM);
                
                // 等待一秒让进程退出
                for (int i = 0; i < 10; ++i) {
                    if (waitpid(child_pid, &status, WNOHANG) == child_pid) {
                        process_terminated = true;
                        break;
                    }
                    usleep(100000); 
                }
                
                // 如果还没退出，向整个进程组强制终止
                if (!process_terminated) {
                    kill(-child_pid, SIGKILL);
                    waitpid(child_pid, &status, 0);
                    process_terminated = true;
                }
                
                return 124; // 超时退出码
            }
            
            // 检查进程是否自然完成
            if (waitpid(child_pid, &status, WNOHANG) == child_pid) {
                process_terminated = true;
                return WEXITSTATUS(status);
            }
            
            // 短暂休眠后继续检查
            usleep(50000); // 50ms
        }
    }
    return -1;
}

std::string CSEProver::extractProblemName(const std::string& problem_path) {
    // 找到最后一个路径分隔符
    size_t last_slash = problem_path.find_last_of("/\\");
    std::string filename;
    if (last_slash != std::string::npos) {
        filename = problem_path.substr(last_slash + 1);
    } else {
        filename = problem_path;
    }
    
    // 只移除TPTP文件扩展名(.p, .cnf, .tptp等)，保留文件名中的其他点号
    // 检查是否以已知的TPTP扩展名结尾
    if (filename.size() >= 2 && filename.substr(filename.size() - 2) == ".p") {
        return filename.substr(0, filename.size() - 2);
    } else if (filename.size() >= 4 && filename.substr(filename.size() - 4) == ".cnf") {
        return filename.substr(0, filename.size() - 4);
    } else if (filename.size() >= 5 && filename.substr(filename.size() - 5) == ".tptp") {
        return filename.substr(0, filename.size() - 5);
    }
    
    // 如果没有匹配的扩展名，返回原文件名
    return filename;
}

std::string CSEProver::buildOutputDirectory(const std::string& problem_name) {

    
    // 辅助函数：在给定的基础输出目录中查找特定于问题的子目录
    auto find_problem_dir_in_base = [&](const std::string& base_output_path) -> std::string {

        
        // 首先检查基础的output目录是否存在
        if (!file_exists(base_output_path)) {

            return "";
        }
        
        // 使用find命令查找匹配的子目录
        std::string find_cmd = "find \"" + base_output_path + "\" -maxdepth 1 -type d -name \"" + problem_name + "*\" 2>/dev/null | head -1";
        
        FILE* pipe = popen(find_cmd.c_str(), "r");
        if (pipe) {
            char buffer[1024];
            if (fgets(buffer, sizeof(buffer), pipe) != nullptr) {
                std::string found_dir(buffer);

                // 移除换行符
                if (!found_dir.empty() && found_dir.back() == '\n') {
                    found_dir.pop_back();
                }
                pclose(pipe);
                // 再次确认找到的目录确实存在
                if (!found_dir.empty() && file_exists(found_dir)) {
                    return found_dir;
                }
            }
            pclose(pipe);
        }
        return ""; // 未找到
    };

    // 1. 优先在当前工作目录下的output中寻找
    std::string work_dir_output = work_directory + "/output";
    std::string found_path = find_problem_dir_in_base(work_dir_output);
    if (!found_path.empty()) {
        return found_path;
    }

    // 2. 如果没找到，且CSE目录与工作目录不同，则在CSE可执行文件目录下的output中寻找
    if (cse_base_directory != work_directory) {
        std::string cse_dir_output = cse_base_directory + "/output";
        found_path = find_problem_dir_in_base(cse_dir_output);
        if (!found_path.empty()) {
            return found_path;
        }
    }

    // 3. 如果都找不到，返回默认路径（基于当前工作目录），CSE将在这里创建新目录
    std::string default_path = work_dir_output + "/" + problem_name + "_short+eq_rstfp";
    return default_path;
}

bool CSEProver::runForLemmas(const std::string& problem_path,
                            const std::string& output_log_path,
                            int time_limit_seconds,
                            int memory_limit_mb) {
    (void)output_log_path; // CSE有自己的输出机制，不使用这个参数
    (void)memory_limit_mb; // 暂时不支持内存限制
    
    // 设置当前问题
    setCurrentProblem(problem_path);
    std::string problem_name = current_problem_name;
    
    // 构建输出目录路径
    std::string output_dir = buildOutputDirectory(problem_name);
    
    // 构建CSE命令参数：/path/to/problem.p work/dir time
    std::vector<std::string> args;
    args.push_back(executable_path);
    args.push_back(problem_path);          // 第一项：问题路径
    args.push_back(work_directory);        // 第二项：工作目录
    args.push_back(std::to_string(time_limit_seconds)); // 第三项：时间限制
    

    
    // 执行CSE进程
    int result = executeCSEProcess(args, time_limit_seconds + 10); // 给额外10秒的缓冲时间
    
    // 检查执行结果
    if (result == 0) {
        return true;
    } else if (result == 124) {
        return true; // 超时也认为是可接受的
    } else if (result == 130) {
        return true; // 提前停止也是正常情况
    } else {
        return false;
    }
}

std::vector<Lemma> CSEProver::parseLogForLemmas(const std::string& log_file_path) {
    (void)log_file_path; // CSE不使用日志文件，而是解析输出目录
    
    // 使用存储的问题名称
    std::string problem_name = current_problem_name;
    
    if (problem_name == "unknown") {
        // 如果没有设置问题名称，尝试从log_file_path中提取
        size_t last_slash = log_file_path.find_last_of("/\\");
        if (last_slash != std::string::npos) {
            std::string filename = log_file_path.substr(last_slash + 1);
            size_t underscore_pos = filename.find("_CSE_");
            if (underscore_pos != std::string::npos) {
                problem_name = filename.substr(0, underscore_pos);
            } else {
                // 尝试其他模式
                underscore_pos = filename.find("_cse_");
                if (underscore_pos != std::string::npos) {
                    problem_name = filename.substr(0, underscore_pos);
                }
            }
        }
        
        // 如果仍然是unknown，使用默认值
        if (problem_name == "unknown") {
            // 使用默认值
        }
    }
    
    // 构建CSE输出目录路径
    std::string output_dir = buildOutputDirectory(problem_name);
    
    // 使用格式转换器解析CSE输出
    std::vector<Lemma> lemmas = LemmaFormatConverter::parseCSEOutputDirectory(output_dir);
    
    return lemmas;
}

bool CSEProver::runForProof(const std::string& problem_path,
                           const std::string& result_file_path,
                           int time_limit_seconds,
                           const std::string& mode_options,
                           int memory_limit_mb) {
    (void)problem_path;     // 消除未使用参数警告
    (void)result_file_path;
    (void)time_limit_seconds;
    (void)mode_options;
    (void)memory_limit_mb;
    
    return false;
}

std::string CSEProver::checkProofResult(const std::string& result_file_path) {
    (void)result_file_path; // 消除未使用参数警告
    
    return "$SZS Status Unknown";
}

std::string CSEProver::getName() const {
    return "CSE";
}

bool CSEProver::checkProofFoundDuringLemmaGeneration(const std::string& cse_log_file) {
    // 改为检测 .out 文件中的 $false 子句，而不是日志文件中的中文信息
    // 首先获取问题名称以构建输出目录路径
    std::string problem_name = current_problem_name;
    
    if (problem_name == "unknown") {
        // 如果问题名称未知，从日志文件路径中提取
        size_t pos = cse_log_file.find("cse_output_");
        if (pos != std::string::npos) {
            size_t start = pos + 11; // "cse_output_" 的长度
            size_t end = cse_log_file.find(".log", start);
            if (end != std::string::npos) {
                problem_name = cse_log_file.substr(start, end - start);
            }
        }
    }
    
    if (problem_name == "unknown") {
        return false;
    }
    
    // 构建 CSE 输出目录路径
    std::string output_dir = buildOutputDirectory(problem_name);
    
    // 查找 .out 文件
    std::string cse_output_file;
    std::string find_cmd = "find \"" + output_dir + "\" -maxdepth 1 -name \"*.out\" 2>/dev/null | head -1";
    
    FILE* pipe = popen(find_cmd.c_str(), "r");
    if (pipe) {
        char buffer[1024];
        if (fgets(buffer, sizeof(buffer), pipe) != nullptr) {
            cse_output_file = std::string(buffer);
            // 移除换行符
            if (!cse_output_file.empty() && cse_output_file.back() == '\n') {
                cse_output_file.pop_back();
            }
        }
        pclose(pipe);
    }
    
    // 如果没找到，使用默认路径
    if (cse_output_file.empty()) {
        cse_output_file = output_dir + "/" + problem_name + "_short+eq_rstfp.out";
    }
    
    // 检查 .out 文件是否包含 $false 子句
    std::ifstream file(cse_output_file);
    if (!file.is_open()) {
        return false;
    }
    
    std::string line;
    while (std::getline(file, line)) {
        // 检查是否包含 $false 子句
        // 格式类似：cnf(i_0_1057, plain, ($false), inference(...), ['proof']).
        if (line.find("cnf(") == 0 && 
            line.find("($false)") != std::string::npos && 
            line.find(").") != std::string::npos) {
            file.close();
            return true;
        }
    }
    
    file.close();
    return false;
}

bool CSEProver::runLemmasWithDynamicCheck(const std::string& problem_path,
                                         int base_time_limit,
                                         int check_interval,
                                         int max_total_time,
                                         int memory_limit_mb) {
    (void)memory_limit_mb; // 消除未使用参数警告
    
    std::string problem_name = current_problem_name;
    if (problem_name == "unknown" || problem_name.empty()) {
        // 只有在没有设置正确名称时才从路径提取 (作为后备)
        problem_name = extractProblemName(problem_path);
    }
    
    std::string output_dir = buildOutputDirectory(problem_name);
    
    // 创建CSE输出日志文件
    std::string cse_log_file = "cse_output_" + problem_name + ".log";
    
    // 构建CSE命令参数：/path/to/problem.p work/dir time
    std::vector<std::string> args;
    args.push_back(executable_path);
    args.push_back(problem_path);
    args.push_back(work_directory);
    args.push_back(std::to_string(max_total_time)); // 使用最大时间作为CSE的时间限制
    
    // 构建命令字符串
    std::string command_str;
    for (size_t i = 0; i < args.size(); ++i) {
        if (i > 0) command_str += " ";
        command_str += args[i];
    }
    
    // 使用fork启动CSE进程
    pid_t cse_pid = fork();
    if (cse_pid == -1) {
        return false;
    }
    
    if (cse_pid == 0) {
        // 子进程：执行CSE
        // 重定向CSE的输出到日志文件
        int fd = open(cse_log_file.c_str(), O_WRONLY | O_CREAT | O_TRUNC, 0644);
        if (fd != -1) {
            dup2(fd, STDOUT_FILENO);
            dup2(fd, STDERR_FILENO);
            close(fd);
        }
        
        std::vector<char*> argv_exec;
        for (const auto& arg : args) {
            argv_exec.push_back(const_cast<char*>(arg.c_str()));
        }
        argv_exec.push_back(nullptr);
        
        execv(executable_path.c_str(), argv_exec.data());
        
        // 如果execv失败
        exit(1);
    } else {
        auto start_time = std::chrono::high_resolution_clock::now();
        bool found_sufficient_lemmas = false;
        bool process_terminated = false;
        
        // 等待基础时间，但期间也检查CSE输出和提前停止标志
        for (int i = 0; i < base_time_limit; ++i) {
            if (proof_found.load()) {
                found_sufficient_lemmas = true;
                break;
            }
            
            // 每隔check_interval秒检查一次CSE输出
            if (i > 0 && i % check_interval == 0) {
                if (checkProofFoundDuringLemmaGeneration(cse_log_file)) {
                    proof_found.store(true);
                    proof_finder_id.store(-1);
                    found_sufficient_lemmas = true;
                    break;
                }
            }
            
            sleep(1); // 每秒检查一次
        }
        
        // 开始定期检查.out文件内容
        int last_total_lines = 0;
        
        while (!found_sufficient_lemmas && !process_terminated) {
            auto current_time = std::chrono::high_resolution_clock::now();
            auto elapsed = std::chrono::duration_cast<std::chrono::seconds>(current_time - start_time);
            
            if (proof_found.load()) {
                break;
            }
            
            // 检查总时间是否超限
            if (elapsed.count() >= max_total_time) {
                break;
            }
            
            // 检查进程是否自然结束
            int status;
            if (waitpid(cse_pid, &status, WNOHANG) == cse_pid) {
                process_terminated = true;
                break;
            }
            
            // 检查单一.out文件的CNF子句行数
            int current_total_lines = 0;
            
            // 动态查找CSE输出的.out文件
            std::string cse_output_file;
            
            // 查找目录中的.out文件
            std::string find_cmd = "find \"" + output_dir + "\" -maxdepth 1 -name \"*.out\" 2>/dev/null | head -1";
            FILE* pipe = popen(find_cmd.c_str(), "r");
            if (pipe) {
                char buffer[1024];
                if (fgets(buffer, sizeof(buffer), pipe) != nullptr) {
                    cse_output_file = std::string(buffer);
                    // 移除换行符
                    if (!cse_output_file.empty() && cse_output_file.back() == '\n') {
                        cse_output_file.pop_back();
                    }
                }
                pclose(pipe);
            }
            
            // 如果没找到，使用默认路径
            if (cse_output_file.empty()) {
                cse_output_file = output_dir + "/" + problem_name + "_short+eq_rstfp.out";
            }
            
            // 检查文件是否存在并统计CNF子句行数
            std::ifstream file(cse_output_file);
            if (file.is_open()) {
                std::string line;
                while (std::getline(file, line)) {
                    // 只计算CNF子句行：以cnf(开头且包含).的行
                    if (!line.empty() && 
                        line.find("cnf(") == 0 && 
                        line.find(").") != std::string::npos) {
                        current_total_lines++;
                    }
                }
                file.close();
            }
            
            // 更新进度
            if (current_total_lines != last_total_lines) {
                last_total_lines = current_total_lines;
            }
            
            // 检查CSE是否找到了证明
            if (checkProofFoundDuringLemmaGeneration(cse_log_file)) {
                // CSE找到了证明！设置提前停止标志
                proof_found.store(true);
                proof_finder_id.store(-1); // 使用-1表示CSE在引理生成阶段找到证明
                found_sufficient_lemmas = true;
                break;
            }
            
            // 检查是否达到100行阈值，如果达到则直接终止
            if (current_total_lines >= 100) {
                found_sufficient_lemmas = true;
                break;
            }
            
            // 等待下一次检查
            sleep(check_interval);
        }
        
        // 如果发现了足够的引理或达到时间限制，停止CSE进程
        if (!process_terminated) {
            kill(cse_pid, SIGTERM);
            
            // 给进程一些时间退出
            for (int i = 0; i < 10; ++i) {
                int status;
                if (waitpid(cse_pid, &status, WNOHANG) == cse_pid) {
                    process_terminated = true;
                    break;
                }
                usleep(100000); // 100ms
            }
            
            // 如果仍未退出，强制终止
            if (!process_terminated) {
                kill(cse_pid, SIGKILL);
                waitpid(cse_pid, nullptr, 0);
            }
        }
        
        auto end_time = std::chrono::high_resolution_clock::now();
        auto total_duration = std::chrono::duration_cast<std::chrono::seconds>(end_time - start_time);
        
        // 最终统计单一.out文件的CNF子句数量
        int final_total_lines = 0;
        
        // 动态查找CSE输出的.out文件
        std::string cse_output_file;
        std::string find_cmd = "find \"" + output_dir + "\" -maxdepth 1 -name \"*.out\" 2>/dev/null | head -1";
        FILE* pipe = popen(find_cmd.c_str(), "r");
        if (pipe) {
            char buffer[1024];
            if (fgets(buffer, sizeof(buffer), pipe) != nullptr) {
                cse_output_file = std::string(buffer);
                // 移除换行符
                if (!cse_output_file.empty() && cse_output_file.back() == '\n') {
                    cse_output_file.pop_back();
                }
            }
            pclose(pipe);
        }
        
        // 如果没找到，使用默认路径
        if (cse_output_file.empty()) {
            cse_output_file = output_dir + "/" + problem_name + "_short+eq_rstfp.out";
        }
        
        std::ifstream final_file(cse_output_file);
        if (final_file.is_open()) {
            std::string line;
            while (std::getline(final_file, line)) {
                if (!line.empty() && 
                    line.find("cnf(") == 0 && 
                    line.find(").") != std::string::npos) {
                    final_total_lines++;
                }
            }
            final_file.close();
        }
        
        if (final_total_lines > 0) {
            return true;
        } else {
            return false;
        }
    }
    
    return false;
}