//** 主程序 - 引理增强的并行证明系统
//** 使用模块化设计，依赖于：
//** - lemma_structures: 核心数据结构和基类
//** - parallel_provers: 证明器实现
//** - lemma_dividers: 引理划分器实现

#include "lemma_structures.h"
#include "parallel_provers.h"
#include "lemma_dividers.h"

//** 全局变量，用于跟踪需要清理的中间文件
std::vector<std::string> intermediate_files_to_cleanup;
std::string problem_name_global;
std::string final_prover_name_global;
int num_lemma_sets_global = 0;

//** 清理中间文件函数
void cleanup_intermediate_files() {
    // 1. 清理明确记录的中间文件
    for (const auto& file : intermediate_files_to_cleanup) {
        if (file_exists(file)) {
            std::remove(file.c_str());
        }
    }
    
    // 2. 清理模式匹配的文件
    if (!problem_name_global.empty()) {
        // 清理CSE输出日志文件
        std::string cse_log_pattern = "cse_output_" + problem_name_global + ".log";
        if (file_exists(cse_log_pattern)) {
            std::remove(cse_log_pattern.c_str());
        }
        
        // 清理统一CNF文件
        std::string unified_cnf = "./output/" + problem_name_global + ".cnf";
        if (file_exists(unified_cnf)) {
            std::remove(unified_cnf.c_str());
        }
    }
    
    // 3. 清理结果文件
    if (!final_prover_name_global.empty()) {
        // 基准进程结果文件
        std::string baseline_result = "result_" + final_prover_name_global + "_baseline.txt";
        if (file_exists(baseline_result)) {
            std::remove(baseline_result.c_str());
        }
        
        // 引理集合结果文件
        for (int i = 1; i <= num_lemma_sets_global; ++i) {
            std::string set_result = "result_" + final_prover_name_global + "_set_" + std::to_string(i) + ".txt";
            if (file_exists(set_result)) {
                std::remove(set_result.c_str());
            }
        }
    }
    
    // 4. 清理临时问题文件
    for (int i = 1; i <= num_lemma_sets_global; ++i) {
        std::string temp_problem = "temp_problem_set_" + std::to_string(i) + ".cnf";
        if (file_exists(temp_problem)) {
            std::remove(temp_problem.c_str());
        }
    }
    
    // 5. 清理可能的其他临时文件
    std::string temp_raw_cnf = "temp_raw_cnf_conversion.cnf";
    if (file_exists(temp_raw_cnf)) {
        std::remove(temp_raw_cnf.c_str());
    }
    
    // 6. 清理引理日志文件（Vampire和CSE）
    std::string vampire_log = "initial_lemmas_Vampire.log";
    if (file_exists(vampire_log)) {
        std::remove(vampire_log.c_str());
    }
    
    std::string cse_log = "initial_lemmas_CSE.log";
    if (file_exists(cse_log)) {
        std::remove(cse_log.c_str());
    }
}

//** 主函数
//** 执行流程：
//** 0. 将输入文件转换为纯CNF格式
//** 1. 解析命令行参数，初始化证明器
//** 2. 运行证明器生成初始引理
//** 3. 解析和过滤引理
//** 4. 将引理划分为多个集合
//** 5. 为每个集合创建新的CNF问题文件
//** 6. 并行运行证明器进行最终证明
//** 
//** @param argc 命令行参数数量
//** @param argv 命令行参数数组
//** @return 程序退出状态码
int main(int argc, char* argv[]) {
    // 检查基本参数数量
    if (argc < 2) {
        return 1;
    }

    std::string original_problem = argv[1];
    std::string lemma_prover_path = "";
    std::string final_prover_path = "";
    
    // 默认参数值
    int initial_time_limit = 5;      // 引理生成时间限制（秒）
    int proof_run_time_limit = 300;    // 并行证明时间限制（秒）
    int num_lemma_sets = 7;           // 引理集合数量
    int memory_limit_mb = 0;       // 内存限制（MB），默认2GB
    int global_time_limit = 300;             // 程序总体时间限制（秒），0表示不限制
    std::string lemma_strategy = "new"; // 引理生成策略
    std::string division_strategy = "depth-based"; // 引理划分策略
    bool enable_early_termination = true;  // 提前停止功能
    bool unit_clauses_only = true;         // 单元子句过滤功能
    bool enable_cpu_affinity = true;       // CPU亲和性绑定功能
    
    // 解析命令行参数
    for (int i = 2; i < argc; i++) {
        std::string arg = argv[i];
        
        if (arg == "--lemma-time" && i + 1 < argc) {
            try {
                initial_time_limit = std::stoi(argv[i + 1]);
                if (initial_time_limit <= 0) {
                    return 1;
                }
                i++; // 跳过下一个参数
            } catch (const std::exception& e) {
                return 1;
            }
        }
        else if (arg == "--proof-time" && i + 1 < argc) {
            try {
                proof_run_time_limit = std::stoi(argv[i + 1]);
                if (proof_run_time_limit <= 0) {
                    return 1;
                }
                i++; // 跳过下一个参数
            } catch (const std::exception& e) {
                return 1;
            }
        }
        else if (arg == "--num-sets" && i + 1 < argc) {
            try {
                num_lemma_sets = std::stoi(argv[i + 1]);
                if (num_lemma_sets <= 0) {
                    return 1;
                }
                i++; // 跳过下一个参数
            } catch (const std::exception& e) {
                return 1;
            }
        }
        else if (arg == "--lemma-prover" && i + 1 < argc) {
            lemma_prover_path = argv[i + 1];
            i++; // 跳过下一个参数
        }
        else if (arg == "--final-prover" && i + 1 < argc) {
            final_prover_path = argv[i + 1];
            i++; // 跳过下一个参数
        }
        else if (arg == "--lemma-strategy" && i + 1 < argc) {
            lemma_strategy = argv[i + 1];
            // 验证策略是否有效
            if (lemma_strategy != "new" && lemma_strategy != "active" && lemma_strategy != "passive") {
                return 1;
            }
            i++; // 跳过下一个参数
        }
        else if (arg == "--division-strategy" && i + 1 < argc) {
            division_strategy = argv[i + 1];

            if (division_strategy != "round-robin" && division_strategy != "age-based" && 
                division_strategy != "depth-based" && division_strategy != "depth-balanced") {
                return 1;
            }
            i++; // 跳过下一个参数
        }
        else if (arg == "--unit-clauses-only") {
            // 启用单元子句过滤
            unit_clauses_only = true;
        }
        else if (arg == "--early-termination") {
            // 启用提前停止
            enable_early_termination = true;
        }
        else if (arg == "--cpu-affinity") {
            // 启用CPU亲和性绑定
            enable_cpu_affinity = true;
        }
        else if (arg == "--memory-limit" && i + 1 < argc) {
            try {
                memory_limit_mb = std::stoi(argv[i + 1]);
                if (memory_limit_mb < 0) {
                    return 1;
                }
                i++; // 跳过下一个参数
            } catch (const std::exception& e) {
                return 1;
            }
        }
        else if (arg == "--global-time-limit" && i + 1 < argc) {
            try {
                global_time_limit = std::stoi(argv[i + 1]);
                if (global_time_limit < 0) {
                    return 1;
                }
                i++; // 跳过下一个参数
            } catch (const std::exception& e) {
                return 1;
            }
        }
        else if (arg == "--help" || arg == "-h") {
            return 0;
        }
        else {
        }
    }
    
    // 验证必需参数
    if (lemma_prover_path.empty()) {
        return 1;
    }
    
    if (final_prover_path.empty()) {
        return 1;
    }
    
    // 从路径自动推断证明器类型
    auto inferProverType = [](const std::string& path) -> std::string {
        if (path.find("vampire") != std::string::npos) {
            return "vampire";
        } else if (path.find("eprover") != std::string::npos) {
            return "eprover";
        } else if (path.find("cse") != std::string::npos) {
            return "cse";
        } else {
            // 默认假设为vampire
            return "vampire";
        }
    };
    
    std::string lemma_prover_type = inferProverType(lemma_prover_path);
    std::string final_prover_type = inferProverType(final_prover_path);
    
    // 确定output目录的基础路径
    std::string output_base_dir = "."; // 默认为当前目录
    
    // 如果引理证明器是CSE，使用CSE可执行文件所在的目录
    if (lemma_prover_type == "cse") {
        if (!lemma_prover_path.empty()) {
            size_t last_slash_pos = lemma_prover_path.find_last_of("/\\");
            if (last_slash_pos != std::string::npos) {
                output_base_dir = lemma_prover_path.substr(0, last_slash_pos);
            }
        }
    }
    
    // 检查并创建output目录
    std::string output_dir = output_base_dir + "/output";
    if (!file_exists(output_dir)) {
        mkdir(output_dir.c_str(), 0755);
    }
    
    // 记录程序总体开始时间
    auto program_start_time = std::chrono::high_resolution_clock::now();
    
    // 提取问题名称（不含路径和后缀）
    std::string problem_name = original_problem;
    size_t last_slash = problem_name.find_last_of("/\\");
    if (last_slash != std::string::npos) {
        problem_name = problem_name.substr(last_slash + 1);
    }
    

    
    // 只移除TPTP文件扩展名.p, .cnf，保留文件名中的其他点号
    if (problem_name.size() >= 2 && problem_name.substr(problem_name.size() - 2) == ".p") {
        problem_name = problem_name.substr(0, problem_name.size() - 2);
    } else if (problem_name.size() >= 4 && problem_name.substr(problem_name.size() - 4) == ".cnf") {
        problem_name = problem_name.substr(0, problem_name.size() - 4);
    }
    

    
    // 初始化全局变量用于清理
    problem_name_global = problem_name;
    num_lemma_sets_global = num_lemma_sets;
    
    // 输出开始证明信息
    std::cout << "# start to proof: " << problem_name << std::endl;
    
    // 全局时间检查函数
    auto check_global_timeout = [&program_start_time, &global_time_limit]() -> bool {
        if (global_time_limit <= 0) return false; // 不限制时间
        auto current_time = std::chrono::high_resolution_clock::now();
        auto elapsed = std::chrono::duration_cast<std::chrono::seconds>(current_time - program_start_time);
        return elapsed.count() >= global_time_limit;
    };
    
    // 提前清理检查函数 - 在超时前1秒开始清理
    auto check_cleanup_time = [&program_start_time, &global_time_limit]() -> bool {
        if (global_time_limit <= 1) return false; // 如果总时间少于等于1秒，不提前清理
        auto current_time = std::chrono::high_resolution_clock::now();
        auto elapsed = std::chrono::duration_cast<std::chrono::seconds>(current_time - program_start_time);
        return elapsed.count() >= (global_time_limit - 1);
    };
    
    // 清理状态跟踪
    std::atomic<bool> cleanup_started(false);
    
    // 定义统一的CNF文件路径（放在output目录中）
    std::string unified_cnf_file = output_dir + "/" + problem_name + ".cnf";
    std::string temp_raw_cnf_file = "temp_raw_cnf_conversion.cnf";
    
    // 统一转换处理：不区分输入文件格式，都通过转换器处理
    // 这样确保总是生成_full.cnf文件和简化的.cnf文件，保持处理流程一致
    int cnf_clause_count = 0;
    
    // 根据引理生成证明器类型选择转换工具，保持一致性
    if (lemma_prover_type == "vampire") {
        // 使用Vampire进行转换（与引理生成保持一致）
        VampireProver vampire_for_cnf(lemma_prover_path, "");
        if (!vampire_for_cnf.convertToCNF(original_problem, temp_raw_cnf_file, global_time_limit)) {
            cleanup_intermediate_files();
            return 1;
        }
        cnf_clause_count = vampire_for_cnf.cleanCNFFile(temp_raw_cnf_file, unified_cnf_file);
    } else {
        // 对于CSE或其他情况，使用E-Prover进行转换（更稳定）
        EProver eprover_for_cnf(final_prover_type == "eprover" ? final_prover_path : lemma_prover_path);
        if (!eprover_for_cnf.convertToCNF(original_problem, temp_raw_cnf_file, global_time_limit)) {
            cleanup_intermediate_files();
            return 1;
        }
        cnf_clause_count = eprover_for_cnf.cleanCNFFile(temp_raw_cnf_file, unified_cnf_file);
    }
    
    if (cnf_clause_count <= 0) {
        cleanup_intermediate_files();
        return 1;
    }
    
    // 删除临时文件
    if (file_exists(temp_raw_cnf_file)) {
        std::remove(temp_raw_cnf_file.c_str());
    }
    
    // 验证证明器兼容性
    if (lemma_prover_type == "eprover") {
        cleanup_intermediate_files();
        return 1;
    }
    
    if (final_prover_type == "cse") {
        cleanup_intermediate_files();
        return 1;
    }
    
    std::unique_ptr<Prover> prover;

    if (lemma_prover_type == "vampire") {
        // 根据策略获取相应的引理生成选项
        std::string lemma_options = VampireProver::getRecommendedLemmaOptions(lemma_strategy);
        prover = make_unique<VampireProver>(lemma_prover_path, lemma_options);
    }
    else if (lemma_prover_type == "cse") {
        // 提取CSE可执行文件的目录
        std::string cse_executable_dir = ".";
        if (!lemma_prover_path.empty()) {
            size_t last_slash_pos = lemma_prover_path.find_last_of("/\\");
            if (last_slash_pos != std::string::npos) {
                cse_executable_dir = lemma_prover_path.substr(0, last_slash_pos);
            }
        }
        // 修改：使工作目录和CSE所在目录相同
        prover = make_unique<CSEProver>(lemma_prover_path, cse_executable_dir, cse_executable_dir);
    }
    else {
        std::cerr << "Error 101: Unsupported lemma prover type '" << lemma_prover_type << "'. Exiting." << std::endl;
        cleanup_intermediate_files();
        return 1;
    }

    // 在任何处理之前立即启动基准进程
    std::vector<std::future<void>> futures; // 使用future管理异步进程
    std::vector<ProcessResult*> process_results; // 存储所有进程结果
    
    // 确定最终证明器名称（用于文件命名）
    std::string final_prover_name;
    if (final_prover_type == "eprover") {
        final_prover_name = "EProver";
    } else if (final_prover_type == "vampire") {
        final_prover_name = "Vampire";
    } else {
        final_prover_name = prover->getName(); // 默认使用引理生成证明器名称
    }
    
    // 初始化全局变量用于清理
    final_prover_name_global = final_prover_name;
    
    // 添加基准进程：使用转换后的CNF文件（不含引理）
    std::string baseline_result_file = "result_" + final_prover_name + "_baseline.txt";
    ProcessResult* baseline_result = new ProcessResult(0, unified_cnf_file, baseline_result_file); // 使用统一CNF文件
    
    process_results.push_back(baseline_result);
    all_processes.push_back(baseline_result); // 添加到提前停止列表
    
    // 立即启动基准进程（与引理生成完全并行）
    std::string proof_run_mode_options;
    if (final_prover_type == "vampire") {
        proof_run_mode_options = "--mode casc --cores 1"; // Vampire的证明运行选项
    } else if (final_prover_type == "eprover") {
        proof_run_mode_options = "--auto --proof-object --output-level=2"; // 使用--auto替代--auto-schedule
    } else {
        proof_run_mode_options = "--mode casc --cores 1"; // 默认使用Vampire选项
    }
    
    futures.emplace_back(std::async(std::launch::async, 
                                   ::run_prover_as_process,
                                   final_prover_path,
                                   baseline_result,
                                   proof_run_time_limit,
                                   proof_run_mode_options,
                                   memory_limit_mb,
                                   enable_early_termination,
                                   enable_cpu_affinity, // 基准进程现在也使用CPU亲和性设置
                                   final_prover_type)); // 基准进程使用最终证明器类型
    
    // 文件路径定义
    const std::string initial_log_file = "initial_lemmas_" + prover->getName() + ".log";
    const std::string temp_problem_dir_prefix = "temp_problem_set_";
    
    // 记录需要清理的中间文件
    intermediate_files_to_cleanup.push_back(initial_log_file);
    intermediate_files_to_cleanup.push_back(baseline_result_file);
    
    // CNF转换处理
    std::string problem_file_for_lemma_generation = unified_cnf_file; // 引理生成使用统一CNF文件
    
    // 检查用户是否通过命令行明确指定了时间
    bool user_specified_time = false;
    for (int i = 3; i < argc; i++) {
        if (std::string(argv[i]) == "--lemma-time") {
            user_specified_time = true;
            break;
        }
    }
    
    if (user_specified_time) {
    } else {
        // 使用自适应时间调整
        initial_time_limit = calculate_adaptive_time_limit(cnf_clause_count, 0);
    }
    
    
    auto stage1_start = std::chrono::high_resolution_clock::now();
    
    // 检查是否支持引理生成
    if (lemma_prover_type == "eprover") {
        
        // 等待基准进程完成
        futures[0].wait(); // 等待基准进程
        
        // 显示基准进程结果
        ProcessResult* baseline_result = process_results[0];
        
        // 在清理资源前进行孤儿进程检测和清理
        detect_and_cleanup_orphan_processes(process_results);
        
        // 清理资源
        for (auto* result : process_results) {
            delete result;
        }
        all_processes.clear();
        
        // 清理中间文件
        cleanup_intermediate_files();
        
        // 返回适当的退出码
        return baseline_result->found_proof ? 0 : 1;
    }
    
    // 使用动态检查机制生成引理（基准进程已在之前启动）
    VampireProver* vampire_prover_dynamic = dynamic_cast<VampireProver*>(prover.get());
    CSEProver* cse_prover_dynamic = dynamic_cast<CSEProver*>(prover.get());
    
    if (vampire_prover_dynamic) {
        // Vampire使用动态检查：基础等待时间为initial_time_limit，每10秒检查一次，最大总时间为120秒
        int max_total_time = std::max(120, initial_time_limit * 2); // 至少120秒，或初始时间的2倍
        vampire_prover_dynamic->runLemmasWithDynamicCheck(
            problem_file_for_lemma_generation, initial_log_file, initial_time_limit, 10, max_total_time, memory_limit_mb);
    } else if (cse_prover_dynamic) {
        // 确保CSE知道当前问题名称
        cse_prover_dynamic->setCurrentProblem(original_problem);
        
        // CSE使用动态检查：基础等待时间为initial_time_limit，每5秒检查一次，最大总时间为120秒
        int max_total_time = std::max(120, initial_time_limit * 2); // 至少120秒，或初始时间的2倍
        bool cse_success = cse_prover_dynamic->runLemmasWithDynamicCheck(
            problem_file_for_lemma_generation, initial_time_limit, 5, max_total_time, memory_limit_mb);
        
        if (!cse_success) {
            // CSE执行失败或未生成引理，但程序继续执行引理解析流程
            // 这种情况下，parseLogForLemmas可能返回空列表，会触发Warning 104
        }
        
        // 检查CSE是否在引理生成过程中找到了证明
        std::string problem_name = cse_prover_dynamic->extractProblemName(original_problem);
        std::string cse_log_file = "cse_output_" + problem_name + ".log";
        if (cse_prover_dynamic->checkProofFoundDuringLemmaGeneration(cse_log_file)) {
            // CSE找到了证明！
            // 生成详细的证明输出（CSE找到证明时没有引理，只有基本CNF转换）
            std::vector<Lemma> empty_lemmas; // CSE在引理生成期间找到证明时没有额外引理
            
            // 计算程序总体运行时间
            auto current_time = std::chrono::high_resolution_clock::now();
            auto program_duration = std::chrono::duration_cast<std::chrono::milliseconds>(current_time - program_start_time);
            double program_total_time = program_duration.count() / 1000.0;
            
            generate_proof_output(
                original_problem,
                unified_cnf_file,
                empty_lemmas,
                cse_log_file, // CSE的日志文件作为结果文件
                true, // 这是"基准"类型的证明（CSE直接找到）
                "CSE",
                program_total_time,
                problem_name // 传递问题名称用于CSE特殊处理
            );
            
            if (enable_early_termination) {
                // 设置提前停止标志
                proof_found.store(true);
                proof_finder_id.store(-1); // 使用-1表示CSE在引理生成阶段找到证明
                
                // 强制终止基准进程
                ProcessResult* baseline_result = process_results[0];
                if (baseline_result && baseline_result->vampire_process_id > 0) {
                    kill(baseline_result->vampire_process_id, SIGTERM);
                    usleep(100000); 
                    kill(baseline_result->vampire_process_id, SIGKILL); // 强制终止
                }
                
                // 等待基准进程完成资源清理
                futures[0].wait();
            } else {
                // 如果没有启用提前停止，仍然等待基准进程完成
                futures[0].wait();
            }
            
            // 在清理资源前进行孤儿进程检测和清理
            detect_and_cleanup_orphan_processes(process_results);
            
            // 清理资源
            for (auto* result : process_results) {
                delete result;
            }
            all_processes.clear();
            
            // 清理中间文件
            cleanup_intermediate_files();
            
            return 0; // 证明成功
        }
        
        // 注意：CSE引理的解析和合并将在后续的标准流程中进行（解析→过滤→划分→合并）
    } else {
        // 回退到原始方法（其他证明器）
        if (!prover->runForLemmas(problem_file_for_lemma_generation, initial_log_file, initial_time_limit, memory_limit_mb)) {
        }
    }
    
    auto stage1_end = std::chrono::high_resolution_clock::now();
    auto stage1_duration = std::chrono::duration_cast<std::chrono::milliseconds>(stage1_end - stage1_start);
    double stage1_time = stage1_duration.count() / 1000.0;
    (void)stage1_time; // 消除未使用变量警告
    
    
    // 检查引理生成过程中是否已经找到证明
    VampireProver* vampire_prover_check = dynamic_cast<VampireProver*>(prover.get());
    if (vampire_prover_check && vampire_prover_check->checkProofFoundDuringLemmaGeneration(initial_log_file)) {
        // 首先检查是否是基准进程在引理生成期间找到的证明
        if (enable_early_termination && proof_found.load() && proof_finder_id.load() == 0) {
            // 基准进程在引理生成期间找到证明
            std::vector<Lemma> empty_lemmas; // 基准进程找到证明时没有额外引理
            std::string baseline_result_file = "result_" + final_prover_name + "_baseline.txt";
            
            // 计算程序总体运行时间
            auto current_time = std::chrono::high_resolution_clock::now();
            auto program_duration = std::chrono::duration_cast<std::chrono::milliseconds>(current_time - program_start_time);
            double program_total_time = program_duration.count() / 1000.0;
            
            generate_proof_output(
                original_problem,
                unified_cnf_file,
                empty_lemmas,
                baseline_result_file,
                true, // 这是基准进程证明
                final_prover_type == "eprover" ? "E-Prover" : "Vampire",
                program_total_time
            );
            
            // 计算总程序运行时间
            // auto program_end_time = std::chrono::high_resolution_clock::now();
            
            // 等待基准进程完成资源清理
            futures[0].wait();
            
            
            // 在清理资源前进行孤儿进程检测和清理
            detect_and_cleanup_orphan_processes(process_results);
            
            // 清理资源
            for (auto* result : process_results) {
                delete result;
            }
            all_processes.clear();
            
            return 0; // 证明成功
        }
        
        // 计算总程序运行时间
        // auto program_end_time = std::chrono::high_resolution_clock::now();
        
        
        // 主动停止基准进程（无需等待，避免浪费资源）
        if (enable_early_termination) {
            // 设置提前停止标志
            proof_found.store(true);
            proof_finder_id.store(-1); // 使用-1表示引理生成阶段找到证明
            
            // 强制终止基准进程
            ProcessResult* baseline_result = process_results[0];
            if (baseline_result && baseline_result->vampire_process_id > 0) {
                kill(baseline_result->vampire_process_id, SIGTERM);
                usleep(100000); 
                kill(baseline_result->vampire_process_id, SIGKILL); // 强制终止
            }
            
            // 等待基准进程完成资源清理
            futures[0].wait();
        } else {
            // 如果没有启用提前停止，仍然等待基准进程完成
            futures[0].wait();
        }
        
        // 在清理资源前进行孤儿进程检测和清理
        detect_and_cleanup_orphan_processes(process_results);
        
        // 清理资源
        for (auto* result : process_results) {
            delete result;
        }
        all_processes.clear();
        
        // 清理中间文件
        cleanup_intermediate_files();
        
        return 0; // 证明成功
    }

    // 检查基准进程是否已完成证明（提前停止检查）
    if (enable_early_termination && proof_found.load()) {
        // ProcessResult* baseline_result = process_results[0]; // 基准进程是第一个
        // 等待基准进程完成并显示结果
        futures[0].wait();
        
        // 基准进程完成证明
        std::vector<Lemma> empty_lemmas; // 基准进程找到证明时没有额外引理
        std::string baseline_result_file = "result_" + final_prover_name + "_baseline.txt";
        
        // 计算程序总体运行时间
        auto current_time = std::chrono::high_resolution_clock::now();
        auto program_duration = std::chrono::duration_cast<std::chrono::milliseconds>(current_time - program_start_time);
        double program_total_time = program_duration.count() / 1000.0;
        
        generate_proof_output(
            original_problem,
            unified_cnf_file,
            empty_lemmas,
            baseline_result_file,
            true, // 这是基准进程证明
            final_prover_type == "eprover" ? "E-Prover" : "Vampire",
            program_total_time
        );
        
        // 在清理并退出前进行孤儿进程检测和清理
        detect_and_cleanup_orphan_processes(process_results);
        
        // 清理并退出
        for (auto* result : process_results) {
            delete result;
        }
        all_processes.clear();
        
        // 清理中间文件
        cleanup_intermediate_files();
        
        return 0;
    }

    
    // 检查全局超时
    if (check_global_timeout()) {
        auto current_time = std::chrono::high_resolution_clock::now();
        auto elapsed = std::chrono::duration_cast<std::chrono::seconds>(current_time - program_start_time);
        // 在超时退出前进行孤儿进程检测和清理
        detect_and_cleanup_orphan_processes(process_results);
        // 清理中间文件
        cleanup_intermediate_files();
        return 124; // 超时退出码
    }
    
    auto stage2_start = std::chrono::high_resolution_clock::now();
    
    std::vector<Lemma> all_extracted_lemmas = prover->parseLogForLemmas(initial_log_file);
    
    auto stage2_end = std::chrono::high_resolution_clock::now();
    auto stage2_duration = std::chrono::duration_cast<std::chrono::milliseconds>(stage2_end - stage2_start);
    double stage2_time = stage2_duration.count() / 1000.0;
    (void)stage2_time; // 消除未使用变量警告
    
    if (all_extracted_lemmas.empty()) {
        std::cerr << "# Warning 104: No lemmas were extracted from the log file '" << initial_log_file << "'. Continuing with baseline prover only." << std::endl;
        
        // 不立即终止，而是等待基准进程完成或达到全局超时
        
        // 循环等待基准进程完成或全局超时
        while (true) {
            // 检查全局超时
            if (check_global_timeout()) {
                // 在超时退出前进行孤儿进程检测和清理
                detect_and_cleanup_orphan_processes(process_results);
                // 清理中间文件
                cleanup_intermediate_files();
                return 124; // 超时退出码
            }
            
            // 检查基准进程是否完成
            if (futures[0].wait_for(std::chrono::seconds(1)) == std::future_status::ready) {
                // 基准进程完成了
                break;
            }
            
            // 如果启用了提前终止，检查是否有进程找到了证明
            if (enable_early_termination && proof_found.load()) {
                break;
            }
        }
        
        // 等待基准进程完成
        futures[0].wait(); // 确保基准进程完全完成
        
        // 显示基准进程结果
        ProcessResult* baseline_result = process_results[0];
        
        // 如果基准进程找到了证明，生成证明输出
        if (baseline_result->found_proof) {
            std::vector<Lemma> empty_lemmas; // 基准进程找到证明时没有额外引理
            std::string baseline_result_file = "result_" + final_prover_name + "_baseline.txt";
            
            // 计算程序总体运行时间
            auto current_time = std::chrono::high_resolution_clock::now();
            auto program_duration = std::chrono::duration_cast<std::chrono::milliseconds>(current_time - program_start_time);
            double program_total_time = program_duration.count() / 1000.0;
            
            generate_proof_output(
                original_problem,
                unified_cnf_file,
                empty_lemmas,
                baseline_result_file,
                true, // 这是基准进程证明
                final_prover_type == "eprover" ? "E-Prover" : "Vampire",
                program_total_time
            );
        }
        
        // 在清理资源前进行孤儿进程检测和清理
        detect_and_cleanup_orphan_processes(process_results);
        
        // 清理资源
        for (auto* result : process_results) {
            delete result;
        }
        all_processes.clear();
        
        // 清理中间文件
        cleanup_intermediate_files();
        
        // 返回适当的退出码
        return baseline_result->found_proof ? 0 : 1;
    }
    


    auto stage3_start = std::chrono::high_resolution_clock::now();
    
    std::vector<Lemma> selected_lemmas;
    
    // 对于depth-based和depth-balanced策略，使用特殊处理：先不过滤，在划分器内部计算深度后再过滤
    if (division_strategy == "depth-based" || division_strategy == "depth-balanced") {

        
        // 为depth-based策略预处理：清理所有引理内容但保留结构
        selected_lemmas.reserve(all_extracted_lemmas.size());
        for (const auto& lemma : all_extracted_lemmas) {
            // 过滤avatar约束的引理（检查两种情况）
            bool has_avatar_constraint = false;
            
            // 1. 检查avatar_id字段
            if (!lemma.avatar_id.empty()) {
                has_avatar_constraint = true;
            }
            
            // 2. 检查content中是否包含AVATAR约束模式 "<- ("
            if (lemma.content.find("<-") != std::string::npos) {
                has_avatar_constraint = true;
            }
            
            // 跳过有AVATAR约束的引理
            if (has_avatar_constraint) {
                continue;
            }
            
            Lemma cleaned_lemma = lemma;
            
            // 提前清理引理内容，避免CNF生成时出错
            // 使用与to_tptp_cnf()相同的清理逻辑
            std::string& clean_content = cleaned_lemma.content;
            
            // 1. 移除或转义引号
            std::replace(clean_content.begin(), clean_content.end(), '"', '\'');
            
            // 2. 移除可能的控制字符
            clean_content.erase(std::remove_if(clean_content.begin(), clean_content.end(), 
                                             [](unsigned char c) { 
                                                 return c < 32 && c != 9 && c != 10 && c != 13;
                                             }), clean_content.end());
            
            // 3. 处理反斜杠
            size_t pos = 0;
            while ((pos = clean_content.find('\\', pos)) != std::string::npos) {
                clean_content.erase(pos, 1);
            }
            
            // 4. 清理前后空格
            clean_content.erase(0, clean_content.find_first_not_of(" \t\n\r\f\v"));
            clean_content.erase(clean_content.find_last_not_of(" \t\n\r\f\v") + 1);
            
            // 5. 确保内容不为空
            if (clean_content.empty()) {
                clean_content = "true";
            }
            
            // 6. 处理特殊的TPTP关键字冲突
            if (clean_content == "$false" || clean_content == "$true") {
                clean_content = "true";
            }
            
            selected_lemmas.push_back(cleaned_lemma);
        }
        

    } else {
        // 所有其他策略（包括symbol-based）都使用标准的引理过滤
        selected_lemmas = filter_lemmas(all_extracted_lemmas, unit_clauses_only);
    }
    
    auto stage3_end = std::chrono::high_resolution_clock::now();
    auto stage3_duration = std::chrono::duration_cast<std::chrono::milliseconds>(stage3_end - stage3_start);
    double stage3_time = stage3_duration.count() / 1000.0;
    (void)stage3_time; // 消除未使用变量警告
    
    if (division_strategy == "depth-based" || division_strategy == "depth-balanced") {
    } else {
    }
    if (selected_lemmas.empty()) {
        std::cerr << "# Warning 105: No lemmas remained after filtering. Continuing with baseline prover only." << std::endl;
        
        // 不立即终止，而是等待基准进程完成或达到全局超时
        
        // 循环等待基准进程完成或全局超时
        while (true) {
            // 检查全局超时
            if (check_global_timeout()) {
                // 在超时退出前进行孤儿进程检测和清理
                detect_and_cleanup_orphan_processes(process_results);
                // 清理中间文件
                cleanup_intermediate_files();
                return 124; // 超时退出码
            }
            
            // 检查基准进程是否完成
            if (futures[0].wait_for(std::chrono::seconds(1)) == std::future_status::ready) {
                // 基准进程完成了
                break;
            }
            
            // 如果启用了提前终止，检查是否有进程找到了证明
            if (enable_early_termination && proof_found.load()) {
                break;
            }
        }
        
        // 等待基准进程完成
        futures[0].wait(); // 确保基准进程完全完成
        
        // 显示基准进程结果
        ProcessResult* baseline_result = process_results[0];
        
        // 如果基准进程找到了证明，生成证明输出
        if (baseline_result->found_proof) {
            std::vector<Lemma> empty_lemmas; // 基准进程找到证明时没有额外引理
            std::string baseline_result_file = "result_" + final_prover_name + "_baseline.txt";
            
            // 计算程序总体运行时间
            auto current_time = std::chrono::high_resolution_clock::now();
            auto program_duration = std::chrono::duration_cast<std::chrono::milliseconds>(current_time - program_start_time);
            double program_total_time = program_duration.count() / 1000.0;
            
            generate_proof_output(
                original_problem,
                unified_cnf_file,
                empty_lemmas,
                baseline_result_file,
                true, // 这是基准进程证明
                final_prover_type == "eprover" ? "E-Prover" : "Vampire",
                program_total_time
            );
        }
        
        // 在清理资源前进行孤儿进程检测和清理
        detect_and_cleanup_orphan_processes(process_results);
        
        // 清理资源
        for (auto* result : process_results) {
            delete result;
        }
        all_processes.clear();
        
        // 清理中间文件
        cleanup_intermediate_files();
        
        // 返回适当的退出码
        return baseline_result->found_proof ? 0 : 1;
    }

    
    // 检查全局超时
    if (check_global_timeout()) {
        auto current_time = std::chrono::high_resolution_clock::now();
        auto elapsed = std::chrono::duration_cast<std::chrono::seconds>(current_time - program_start_time);
        // 在超时退出前进行孤儿进程检测和清理
        detect_and_cleanup_orphan_processes(process_results);
        // 清理中间文件
        cleanup_intermediate_files();
        return 124; // 超时退出码
    }
    
    // 提前声明变量，避免作用域问题
    double stage4_time = 0.0;
    double stage5_time = 0.0;
    double stage6_time = 0.0; 
    // 消除未使用变量警告
    (void)stage4_time;
    (void)stage5_time;
    (void)stage6_time;
    bool stage6_started = false; 
    auto lemma_enhanced_start_time = std::chrono::high_resolution_clock::now();
    auto stage6_start_time = lemma_enhanced_start_time;
    
    // 检查提前停止标志 - 如果基准进程已经找到证明，跳过引理划分
    if (enable_early_termination && proof_found.load()) {
        // 直接跳转到等待和结果汇总阶段
        goto wait_and_summary;
    }
    
    {
    auto stage4_start = std::chrono::high_resolution_clock::now();
    
    // 创建选定的引理划分器
    std::unique_ptr<LemmaDivider> lemma_divider = createLemmaDivider(division_strategy);
    
    // 使用选定的划分策略
        std::vector<std::vector<Lemma>> lemma_sets;
        
        // 对于depth-based和depth-balanced策略，使用支持unit_clauses_only参数的重载方法
        if (division_strategy == "depth-based") {
            DepthBasedDivider* depth_divider = dynamic_cast<DepthBasedDivider*>(lemma_divider.get());
            if (depth_divider) {
                lemma_sets = depth_divider->divide(selected_lemmas, num_lemma_sets, unit_clauses_only);
            } else {
                lemma_sets = lemma_divider->divide(selected_lemmas, num_lemma_sets);
            }
        } else if (division_strategy == "depth-balanced") {
            DepthBalancedDivider* depth_balanced_divider = dynamic_cast<DepthBalancedDivider*>(lemma_divider.get());
            if (depth_balanced_divider) {
                lemma_sets = depth_balanced_divider->divide(selected_lemmas, num_lemma_sets, unit_clauses_only);
            } else {
                lemma_sets = lemma_divider->divide(selected_lemmas, num_lemma_sets);
            }
        } else {
            // 其他策略使用标准方法
            lemma_sets = lemma_divider->divide(selected_lemmas, num_lemma_sets);
        }
    
    auto stage4_end = std::chrono::high_resolution_clock::now();
    auto stage4_duration = std::chrono::duration_cast<std::chrono::milliseconds>(stage4_end - stage4_start);
        stage4_time = stage4_duration.count() / 1000.0;
    
    
        // 检查提前停止标志 - 如果基准进程已经找到证明，跳过文件创建
        if (enable_early_termination && proof_found.load()) {
            goto wait_and_summary;
        }
        
    auto stage5_start = std::chrono::high_resolution_clock::now();
    
    if (!create_new_cnf_problem_files(unified_cnf_file, lemma_sets, temp_problem_dir_prefix)) {
    }
    
    auto stage5_end = std::chrono::high_resolution_clock::now();
    auto stage5_duration = std::chrono::duration_cast<std::chrono::milliseconds>(stage5_end - stage5_start);
        stage5_time = stage5_duration.count() / 1000.0;
    
    
    
    // 记录引理增强进程的开始时间
        stage6_start_time = std::chrono::high_resolution_clock::now();
        stage6_started = true;  
    
    // 启动引理增强的并行进程
    for (int i = 0; i < num_lemma_sets; ++i) {
        // 检查提前停止标志
        if (enable_early_termination && proof_found.load()) {
            break;
        }
        
        // 确保lemma_sets[i]实际存在并且非空
        if (static_cast<size_t>(i) >= lemma_sets.size() || lemma_sets[i].empty()) {
            continue;
        }
        std::string problem_file = temp_problem_dir_prefix + std::to_string(i + 1) + ".cnf";
        // 检查问题文件是否存在，因为create_new_cnf_problem_files可能会失败
        if (!file_exists(problem_file)) {
            continue;
        }
        std::string result_file = "result_" + final_prover_name + "_set_" + std::to_string(i + 1) + ".txt";

        // 创建进程结果对象
        ProcessResult* process_result = new ProcessResult(i + 1, problem_file, result_file);
        process_results.push_back(process_result);
        all_processes.push_back(process_result); // 添加到提前停止列表

        // 使用std::async启动异步进程
        futures.emplace_back(std::async(std::launch::async, 
                                       ::run_prover_as_process,
                                       final_prover_path,
                                       process_result,
                                       proof_run_time_limit,
                                       proof_run_mode_options,
                                       memory_limit_mb,
                                       enable_early_termination,
                                       enable_cpu_affinity,
                                       final_prover_type));
    }
    }

wait_and_summary:
    // 等待所有异步进程完成
    if (enable_early_termination) {
        
        while (true) {
            bool all_completed = true;
            
            // 优先检查提前清理时间 - 在超时前1秒开始清理
            if (!cleanup_started.load() && check_cleanup_time()) {
                cleanup_started.store(true);
                
                // 记录清理开始时间
                auto cleanup_start_time = std::chrono::high_resolution_clock::now();
                auto elapsed = std::chrono::duration_cast<std::chrono::seconds>(cleanup_start_time - program_start_time);
                
                // 开始清理中间文件（在后台异步执行，不阻塞主循环）
                std::thread cleanup_thread([&]() {
                    cleanup_intermediate_files();
                });
                cleanup_thread.detach(); // 让清理线程在后台运行
                
                // 设置提前停止标志，开始关闭进程
                proof_found.store(true);
                proof_finder_id.store(-998); // 使用-998表示提前清理阶段
            }
            
            // 检查全局超时
            if (check_global_timeout()) {
                auto current_time = std::chrono::high_resolution_clock::now();
                auto elapsed = std::chrono::duration_cast<std::chrono::seconds>(current_time - program_start_time);
                
                // 设置提前停止标志，强制终止所有进程
                proof_found.store(true);
                proof_finder_id.store(-999); // 使用特殊ID表示超时终止
                break;
            }
            
            // 检查所有future是否完成
            for (auto& future : futures) {
                if (future.wait_for(std::chrono::milliseconds(50)) != std::future_status::ready) {
                    all_completed = false;
                }
            }
            
            // 如果所有进程都完成了，或者发现了证明，就退出等待
            if (all_completed || proof_found.load()) {
                break;
            }
            
            // 缩短休眠时间，更频繁地检查超时
            std::this_thread::sleep_for(std::chrono::milliseconds(20));
        }
        
        // 确保所有future都已完成（提前停止的进程也需要完成资源清理）
        // 对于全局超时情况，限制等待时间避免无限等待
        int finder_id = proof_finder_id.load();
        bool timeout_related = (finder_id == -999 || finder_id == -998);
        
        if (timeout_related) {
            // 超时相关情况（包括提前清理和真正超时），限制等待时间
            int max_wait_seconds = (finder_id == -998) ? 2 : 1; // 提前清理给稍多时间
            
            for (auto& future : futures) {
                if (future.wait_for(std::chrono::seconds(max_wait_seconds)) == std::future_status::timeout) {
                    // 如果超时后仍未完成，不再等待，让系统自行清理
                    break;
                }
            }
        } else {
            // 正常情况下等待所有进程完成
            for (auto& future : futures) {
                future.wait();
            }
        }
    } else {
        // 等待所有进程完成
        for (auto& future : futures) {
            future.wait(); // 等待每个进程完成
        }
    }
    
    // 计算引理增强进程的执行时间
    auto total_end_time = std::chrono::high_resolution_clock::now();
    
    // 计算正确的阶段6时间
    if (stage6_started) {
        // 如果阶段6真正开始了，计算从阶段6开始到结束的时间
        auto stage6_duration = std::chrono::duration_cast<std::chrono::milliseconds>(total_end_time - stage6_start_time);
        stage6_time = stage6_duration.count() / 1000.0;
    } else {
        // 如果阶段6没有开始，时间为0
        stage6_time = 0.0;
    }
    
    // 计算从阶段4开始的总时间
    auto total_duration = std::chrono::duration_cast<std::chrono::milliseconds>(total_end_time - lemma_enhanced_start_time);
    double lemma_enhanced_time_seconds = total_duration.count() / 1000.0;
    (void)lemma_enhanced_time_seconds; // 消除未使用变量警告

    
    // 显示提前停止结果
    if (enable_early_termination && proof_found.load()) {
        int finder_id = proof_finder_id.load();
        if (finder_id == -999) {
        } else {
            std::string process_name = (finder_id == 0) ? "基准进程" : ("进程 " + std::to_string(finder_id));
        }
    }

    // 汇总和显示结果
    bool proof_was_found = false;
    bool satisfiable_was_found = false;
    std::string winning_status;
    std::string problem_filename = original_problem;
    size_t filename_last_slash = problem_filename.find_last_of("/\\");
    if (filename_last_slash != std::string::npos) {
        problem_filename = problem_filename.substr(filename_last_slash + 1);
    }
    
    for (const auto& result : process_results) {
        if (result->found_proof) {
            if (result->szs_status.find("Theorem") != std::string::npos || 
                result->szs_status.find("Unsatisfiable") != std::string::npos) {
            proof_was_found = true;
            winning_status = result->szs_status;
            break;
            } else if (result->szs_status.find("Satisfiable") != std::string::npos) {
                satisfiable_was_found = true;
                winning_status = result->szs_status;
                // 不break，继续寻找是否有真正的证明
            }
        }
    }
    
    bool global_timeout_occurred = (enable_early_termination && proof_found.load() && 
                                    (proof_finder_id.load() == -999 || proof_finder_id.load() == -998));

    if (proof_was_found) {
        // 找到获胜的进程以获取结果文件
        std::string winning_result_file;
        std::string winning_problem_file;
        for (const auto& result : process_results) {
            if (result->found_proof && 
                (result->szs_status.find("Theorem") != std::string::npos || 
                 result->szs_status.find("Unsatisfiable") != std::string::npos)) {
                winning_result_file = result->result_file;
                winning_problem_file = result->problem_file;
                break;
            }
        }
        
        // 从获胜进程的问题文件中提取实际使用的引理
        std::vector<Lemma> actual_used_lemmas;
        if (!winning_problem_file.empty()) {
            actual_used_lemmas = extract_lemmas_from_problem_file(winning_problem_file);
        }
        
        // 为引理增强进程生成详细的证明输出
        // 计算程序总体运行时间
        auto program_end_time = std::chrono::high_resolution_clock::now();
        auto program_duration = std::chrono::duration_cast<std::chrono::milliseconds>(program_end_time - program_start_time);
        double program_total_time = program_duration.count() / 1000.0;
        
        generate_proof_output(
            original_problem,
            unified_cnf_file,
            actual_used_lemmas, // 使用获胜进程实际使用的引理
            winning_result_file,
            false, // 这不是基准进程证明
            final_prover_type == "eprover" ? "E-Prover" : "Vampire",
            program_total_time
        );

    } else if (satisfiable_was_found) {
        // 找到Satisfiable结果的进程，也要输出详细的证明路径
        std::string satisfiable_result_file;
        std::string satisfiable_problem_file;
        for (const auto& result : process_results) {
            if (result->found_proof && 
                result->szs_status.find("Satisfiable") != std::string::npos) {
                satisfiable_result_file = result->result_file;
                satisfiable_problem_file = result->problem_file;
                break;
            }
        }
        
        // 从获胜进程的问题文件中提取实际使用的引理
        std::vector<Lemma> actual_used_lemmas;
        if (!satisfiable_problem_file.empty()) {
            actual_used_lemmas = extract_lemmas_from_problem_file(satisfiable_problem_file);
        }
        
        // 计算程序总体运行时间
        auto program_end_time = std::chrono::high_resolution_clock::now();
        auto program_duration = std::chrono::duration_cast<std::chrono::milliseconds>(program_end_time - program_start_time);
        double program_total_time = program_duration.count() / 1000.0;
        
        // 生成详细的证明输出（包括满足性证明路径）
        generate_proof_output(
            original_problem,
            unified_cnf_file,
            actual_used_lemmas,
            satisfiable_result_file,
            satisfiable_problem_file == original_problem, // 如果使用原始问题文件则为基准进程
            final_prover_type == "eprover" ? "E-Prover" : "Vampire",
            program_total_time
        );
    } else if (global_timeout_occurred) {
        std::cout << "CSE_E is exited by timeout" << std::endl;
    } else {
        bool any_timeout = false;
        bool any_satisfiable = false;
        std::string common_status;
        
        for (const auto& result : process_results) {
            if (result->szs_status.find("Timeout") != std::string::npos) {
                any_timeout = true;
                break;
            }
            if (result->szs_status.find("Satisfiable") != std::string::npos) {
                any_satisfiable = true;
                if (common_status.empty()) {
                    common_status = result->szs_status;
        }
            }
        }
        
        if (any_timeout) {
            std::string program_name = argv[0];
            size_t last_slash_prog = program_name.find_last_of("/\\");
            if (last_slash_prog != std::string::npos) {
                program_name = program_name.substr(last_slash_prog + 1);
            }
            std::cout << "CSE_E is exited by timeout" << std::endl;
        } else if (any_satisfiable) {
            // 没有启用提前停止时，所有进程都返回Satisfiable状态
            std::cout << "# SZS status Satisfiable for " << problem_filename << std::endl;
        } else {
            // 检查是否所有进程都正常结束但未找到确定性结果
            bool all_failed_normally = true;
            for (const auto& result : process_results) {
                if (result->found_proof || result->exit_code == -2) { // not found or early terminated
                    all_failed_normally = false;
                    break;
                }
            }
            if (all_failed_normally) {
                 std::cout << "# No proof found" << std::endl;
            }
        }
    }
    
    // 在清理内存前进行孤儿进程检测和清理
    detect_and_cleanup_orphan_processes(process_results);
    
    // 清理动态分配的内存
    for (auto* result : process_results) {
        delete result;
    }
    all_processes.clear(); // 清理提前停止列表
    
    // 计算并显示程序总体运行时间
    auto program_end_time = std::chrono::high_resolution_clock::now();
    auto program_duration = std::chrono::duration_cast<std::chrono::milliseconds>(program_end_time - program_start_time);
    double program_total_time = program_duration.count() / 1000.0;
    (void)program_total_time; // 消除未使用变量警告
    
    // 清理中间文件（如果还没有清理过）
    if (!cleanup_started.load()) {
        cleanup_intermediate_files();
    }
    
    // 根据是否超时返回不同的退出码
    if (global_timeout_occurred) {
        return 124; // 超时退出码
    }

    return 0;
}


//编译指令
//g++ -Wall -std=c++11 -O3 *.cpp -lm -pthread -o lemma_parallel_prover
//cp lemma_parallel_prover ../bin
//./bin/lemma_parallel_prover ./problem_tptp_2024/BIO005+1.p --lemma-prover ./bin/cse --final-prover ./bin/eprover
//使用Vampire生成引理和证明:
//./lemma_parallel_prover ../problem_tptp_2024/SCT170+3.p --lemma-prover ./vampire --final-prover ./vampire --lemma-strategy new --proof-time 300 --num-sets 4 --early-termination --division-strategy depth-based --cpu-affinity  --unit-clauses-only 

//Vampire生成引理，E-Prover证明:
//./lemma_parallel_prover ../problem_tptp_2024/ITP012+4.p --lemma-prover ./vampire --final-prover ./eprover --lemma-strategy new --proof-time 300 --num-sets 4 --early-termination --division-strategy depth-based

//使用CSE生成引理，E-Prover证明:
//./lemma_parallel_prover ./problem_tptp_2023/GEO167+1.p --lemma-prover ./cse --final-prover ./eprover --proof-time 300 --num-sets 4 --early-termination --division-strategy depth-based --cpu-affinity --unit-clauses-only --global-time-limit 300