#ifndef LEMMA_STRUCTURES_H
#define LEMMA_STRUCTURES_H

#include <iostream>
#include <fstream>
#include <string>
#include <vector>
#include <sstream>
#include <algorithm>
#include <regex>
#include <cstdlib>
#include <memory>
#include <map>
#include <chrono>       // 用于时间统计
#include <iomanip>      // 用于时间格式化
#include <limits>       // 用于numeric_limits
#include <climits>      // 用于INT_MAX等常量
#include <future>       // 用于异步进程管理
#include <atomic>       // 用于原子操作
#include <mutex>        // 用于互斥锁
#include <cmath>        // 用于数学函数
#include <set>          // 用于std::set容器
#include <signal.h>     // Unix信号处理
#include <sys/wait.h>   // Unix进程等待
#include <sys/types.h>  // 用于pid_t类型
#include <sys/stat.h>   // 用于mkdir和stat函数
#include <unistd.h>     // Unix系统调用
#include <fcntl.h>      // 用于open和dup2
#include <sched.h>      // 用于CPU亲和性设置
#include <thread>       // 用于线程操作
#include <queue>        // 用于BFS算法

// C++11兼容的make_unique实现
template<typename T, typename... Args>
std::unique_ptr<T> make_unique(Args&&... args) {
    return std::unique_ptr<T>(new T(std::forward<Args>(args)...));
}

// 平台相关的文件存在检查函数
bool file_exists(const std::string& path);

// 进程执行结果结构体
struct ProcessResult {
    int process_id;
    std::string problem_file;
    std::string result_file;
    std::string szs_status;
    double execution_time_seconds;
    bool found_proof;
    int exit_code;
    pid_t vampire_process_id; // Unix进程ID，用于提前停止
    
    ProcessResult(int id, const std::string& prob_file, const std::string& res_file)
        : process_id(id), problem_file(prob_file), result_file(res_file), 
          szs_status("$SZS Status Unknown"), execution_time_seconds(0.0), 
          found_proof(false), exit_code(-1), vampire_process_id(-1) {
    }
};

// 提前停止控制全局变量声明
extern std::atomic<bool> proof_found;      // 是否已找到证明
extern std::atomic<int> proof_finder_id;   // 找到证明的进程ID
extern std::vector<ProcessResult*> all_processes; // 所有进程的引用，用于终止
extern std::mutex early_termination_mutex; // 保护提前停止逻辑的互斥锁

// 辅助结构体，表示一个引理
struct Lemma {
    std::string id;
    std::string role;
    std::string content;

    // 来源信息字段
    std::string derivation_type;  // 推导类型，如"new"或"passive" 
    int clause_number;            // 原始子句编号，如果无法解析则为-1
    std::string source_info;      // 来源信息，如"input"
    std::string avatar_id;        // AVATAR组件ID，如果没有则为空

    // 推导深度相关字段
    std::vector<int> parent_clause_numbers;  // 父子句编号列表
    std::string inference_rule;              // 推理规则名称，如"resolution", "superposition"
    int derivation_depth;                    // 推导深度，-1表示未计算

    // CSE引理专用：保存原始完整的TPTP CNF行（用于输出）
    std::string original_tptp_line;          // 原始的完整cnf(...)行，只有CSE引理才使用

    // 构造函数初始化推导深度
    Lemma() : clause_number(-1), derivation_depth(-1) {}

    //** 将生成的引理转换为TPTP CNF格式的字符串
    //** @return 格式化的TPTP CNF字符串，如"cnf(id, role, (content))."
    std::string to_tptp_cnf() const;
};

// --- Prover 抽象基类 ---
class Prover {
public:
    virtual ~Prover() = default;

    //** 运行证明器以生成引理
    //** @param problem_path 问题文件的路径
    //** @param output_log_path 输出日志文件的路径
    //** @param time_limit_seconds 时间限制（秒）
    //** @param memory_limit_mb 内存限制（MB），0表示不限制
    //** @return 返回值表示命令是否成功启动，不表示证明结果
    virtual bool runForLemmas(const std::string& problem_path,
                              const std::string& output_log_path,
                              int time_limit_seconds,
                              int memory_limit_mb = 0) = 0;

    //** 解析证明器的日志/输出以提取引理
    //** @param log_file_path 日志文件路径
    //** @return 提取到的引理列表
    virtual std::vector<Lemma> parseLogForLemmas(const std::string& log_file_path) = 0;

    //** 运行证明器进行最终证明
    //** @param problem_path 问题文件路径
    //** @param result_file_path 结果文件路径
    //** @param time_limit_seconds 时间限制（秒）
    //** @param mode_options 证明器模式选项
    //** @param memory_limit_mb 内存限制（MB），0表示不限制
    //** @return 返回值表示命令是否成功启动，不表示证明结果
    virtual bool runForProof(const std::string& problem_path,
                             const std::string& result_file_path,
                             int time_limit_seconds,
                             const std::string& mode_options,
                             int memory_limit_mb = 0) = 0;

    //** 检查证明结果，返回SZS状态等
    //** @param result_file_path 结果文件路径
    //** @return SZS状态字符串，如"$SZS Status Theorem"
    virtual std::string checkProofResult(const std::string& result_file_path) = 0;

    //** 获取证明器名称
    //** @return 证明器名称字符串
    virtual std::string getName() const = 0;
};

// --- LemmaDivider 抽象基类 ---
class LemmaDivider {
public:
    virtual ~LemmaDivider() = default;

    //** 将引理集合划分为多个子集
    //** @param lemmas 待划分的引理列表
    //** @param num_sets 划分成的子集数量
    //** @return 划分后的引理子集列表
    virtual std::vector<std::vector<Lemma>> divide(const std::vector<Lemma>& lemmas, int num_sets) = 0;

    //** 获取划分策略的名称
    //** @return 策略名称字符串
    virtual std::string getName() const = 0;

    //** 获取划分策略的描述
    //** @return 策略描述字符串
    virtual std::string getDescription() const = 0;
};

// 引理处理相关工具函数声明

//** 计算引理的推导深度
//** @param lemmas 引理列表（会被修改，添加深度信息）
void calculateDerivationDepths(std::vector<Lemma>& lemmas);

//** 判断是否为单元子句
//** @param content 子句内容
//** @return 如果是单元子句返回true，否则返回false
bool is_unit_clause(const std::string& content);

//** 过滤引理
//** @param all_lemmas 所有引理列表
//** @param unit_clauses_only 是否只保留单元子句
//** @return 过滤后的引理列表
std::vector<Lemma> filter_lemmas(const std::vector<Lemma>& all_lemmas, bool unit_clauses_only = false);

//** 深度相关的引理过滤函数
//** @param lemmas_with_depths 带深度信息的引理列表
//** @param unit_clauses_only 是否只保留单元子句
//** @return 过滤后的引理列表
std::vector<Lemma> filter_lemmas_for_depth_based(const std::vector<Lemma>& lemmas_with_depths, bool unit_clauses_only = false);

//** 创建新的CNF问题文件，包含原始问题和指定的引理集
//** @param cnf_problem_path 原始CNF问题文件路径
//** @param lemma_sets 引理集合向量
//** @param output_file_prefix 输出文件前缀
//** @return 是否成功创建所有文件
bool create_new_cnf_problem_files(const std::string& cnf_problem_path,
                                const std::vector<std::vector<Lemma>>& lemma_sets,
                                const std::string& output_file_prefix);

//** 作为独立进程运行证明器，并支持提前停止
//** @param prover_exec_path 证明器可执行文件路径
//** @param result_ptr 存储结果的指针
//** @param proof_time_limit 证明时间限制（秒）
//** @param proof_mode_options 证明器选项
//** @param memory_limit_mb 内存限制（MB）
//** @param enable_early_termination 是否启用提前停止
//** @param enable_cpu_affinity 是否启用CPU亲和性
//** @param prover_type 证明器类型
void run_prover_as_process(const std::string& prover_exec_path,
                          ProcessResult* result_ptr,
                          int proof_time_limit,
                          const std::string& proof_mode_options,
                          int memory_limit_mb = 0,
                          bool enable_early_termination = false,
                          bool enable_cpu_affinity = false,
                          const std::string& prover_type = "vampire");

//** 根据CNF文件大小自适应计算时间限制
//** @param cnf_clause_count CNF子句数量
//** @param user_specified_time 用户指定的时间（0表示使用自适应）
//** @return 计算得到的时间限制（秒）
int calculate_adaptive_time_limit(int cnf_clause_count, int user_specified_time);

//** 将CSE生成的引理合并到CNF文件中
//** 读取CSE输出目录中的.out文件，将引理转换为CNF格式并添加到基础CNF文件中
//** @param base_cnf_path 基础CNF文件路径
//** @param cse_output_dir CSE输出目录路径
//** @param merged_cnf_path 合并后的CNF文件路径
//** @return 成功合并的引理数量，失败时返回-1
int merge_cse_lemmas_to_cnf(const std::string& base_cnf_path,
                           const std::string& cse_output_dir,
                           const std::string& merged_cnf_path);

// --- 通用引理格式转换器 ---
class LemmaFormatConverter {
public:
    //** 将CSE格式的cnf子句转换为统一的引理格式
    //** 输入：cnf(i_0_13, plain, (content), inference(rule,[],[parents])).
    //** 输出：统一的Lemma结构
    //** @param cse_cnf_line CSE输出的单行cnf子句
    //** @param lemma_counter 引理计数器（用于生成ID）
    //** @return 转换后的Lemma对象，失败时返回空的Lemma
    static Lemma convertCSECnfToLemma(const std::string& cse_cnf_line, int& lemma_counter);
    
    //** 将Vampire格式的引理行转换为统一格式
    //** 输入：[SA] new/active/passive: 123. content [inference_info]
    //** 输出：统一的Lemma结构
    //** @param vampire_line Vampire输出的引理行
    //** @param lemma_counter 引理计数器
    //** @return 转换后的Lemma对象
    static Lemma convertVampireLemmaToUnified(const std::string& vampire_line, int& lemma_counter);
    
    //** 清理引理内容，移除证明器特定的推导信息
    //** 在合并不同证明器的引理时，需要移除具体的推导信息以避免冲突
    //** @param lemma 需要清理的引理对象
    //** @return 清理后的引理对象
    static Lemma cleanLemmaForMerging(const Lemma& lemma);
    
    //** 从文件中读取并转换CSE生成的所有.out文件为统一引理格式
    //** @param cse_output_directory CSE的输出目录路径
    //** @return 转换后的引理列表
    static std::vector<Lemma> parseCSEOutputDirectory(const std::string& cse_output_directory);
    
private:
    //** 解析CSE的cnf子句ID，提取数字部分
    //** 例如：i_0_13 -> 13
    static int extractCSEClauseNumber(const std::string& cse_id);
    
    //** 解析CSE的推理规则信息
    //** 例如：inference(scs_inference,[],[i_0_11,i_0_10]) -> rule="scs_inference", parents=[11,10]
    static void parseCSEInferenceInfo(const std::string& inference_str, 
                                     std::string& rule, 
                                     std::vector<int>& parent_numbers);
};

// --- 孤儿进程检测和清理功能（Linux专用）---

//** 获取指定进程的所有子进程PID
//** @param parent_pid 父进程PID
//** @return 子进程PID列表
std::vector<pid_t> get_child_processes(pid_t parent_pid);

//** 递归获取进程树中的所有进程PID（包括子进程、孙进程等）
//** @param root_pid 根进程PID
//** @return 整个进程树的PID列表（包括根进程）
std::vector<pid_t> get_process_tree(pid_t root_pid);

//** 终止整个进程树（包括所有子进程）
//** @param root_pid 根进程PID
//** @param process_name 进程名称（用于日志）
//** @return 成功终止的进程数量
int terminate_process_tree(pid_t root_pid, const std::string& process_name);

//** 检测并清理程序产生的所有孤儿进程（包括子进程树）
//** @param process_results 进程结果列表
//** @return 检测到的孤儿进程总数
int detect_and_cleanup_orphan_processes(const std::vector<ProcessResult*>& process_results);

//** 检查进程是否正在运行
//** @param pid 进程ID
//** @return 如果进程正在运行返回true，否则返回false
bool is_process_running(pid_t pid);

//** 终止指定进程
//** @param pid 进程ID
//** @param process_name 进程名称（用于日志）
//** @return 如果成功终止返回true，否则返回false
bool terminate_process_gracefully(pid_t pid, const std::string& process_name);

//** 获取进程的命令行信息
//** @param pid 进程ID
//** @return 进程的命令行字符串
std::string get_process_cmdline(pid_t pid);

// --- 证明输出生成功能 ---

//** 生成详细的证明输出
//** 集成CNF转换、引理和推导过程的完整证明输出
//** @param original_problem_path 原始问题文件路径
//** @param cnf_file_path CNF文件路径
//** @param lemmas 引理列表
//** @param prover_result_file 证明器结果文件路径
//** @param is_baseline_proof 是否为基准进程证明
//** @param prover_name 证明器名称
//** @param total_time_seconds 总运行时间（秒）
//** @param problem_name 问题名称（用于CSE特殊处理）
void generate_proof_output(const std::string& original_problem_path,
                          const std::string& cnf_file_path,
                          const std::vector<Lemma>& lemmas,
                          const std::string& prover_result_file,
                          bool is_baseline_proof,
                          const std::string& prover_name,
                          double total_time_seconds,
                          const std::string& problem_name = "");

//** 输出CNF转换部分
//** @param cnf_file_path CNF文件路径
void output_cnf_conversion(const std::string& cnf_file_path, const std::string& original_problem_path);

//** 输出引理部分（CSE格式的引理直接输出，保留原始格式）
//** @param lemmas 引理列表
void output_lemmas(const std::vector<Lemma>& lemmas);

//** 输出推导过程（从E-Prover结果文件中提取）
//** @param prover_result_file 证明器结果文件路径
void output_derivation(const std::string& prover_result_file);

//** 从E-Prover输出中提取推导过程
//** @param result_file 结果文件路径
//** @return 推导过程行列表
std::vector<std::string> extract_derivation_from_eprover(const std::string& result_file);

// 新增函数：输出CSE的.out文件内容
void output_cse_out_file_content(const std::string& problem_name);

// 新增函数：从问题文件中提取引理集合
std::vector<Lemma> extract_lemmas_from_problem_file(const std::string& problem_file_path);

#endif // LEMMA_STRUCTURES_H 
