#include "lemma_structures.h"
#include <cstring>  // 添加这个头文件以使用strerror函数
#include <dirent.h> // 用于目录操作
#include <algorithm> // 用于std::reverse
#include <iomanip> // 用于std::fixed和std::setprecision

// 平台相关的文件存在检查函数实现
#ifdef _WIN32
    #include <windows.h>
    bool file_exists(const std::string& path) {
        DWORD dwAttrib = GetFileAttributes(path.c_str());
        return (dwAttrib != INVALID_FILE_ATTRIBUTES);
    }
#else
    #include <sys/stat.h>
    bool file_exists(const std::string& path) {
        struct stat buffer;
        return (stat(path.c_str(), &buffer) == 0);
    }
#endif

// 提前停止控制全局变量定义
std::atomic<bool> proof_found{false};      // 是否已找到证明
std::atomic<int> proof_finder_id{-1};      // 找到证明的进程ID
std::vector<ProcessResult*> all_processes; // 所有进程的引用，用于终止
std::mutex early_termination_mutex;        // 保护提前停止逻辑的互斥锁

// Lemma 结构体方法实现
std::string Lemma::to_tptp_cnf() const {
    // 清理和验证内容
    std::string clean_content = content;
    
    // 1. 移除或转义引号
    std::replace(clean_content.begin(), clean_content.end(), '"', '\'');
    
    // 2. 移除可能的控制字符
    clean_content.erase(std::remove_if(clean_content.begin(), clean_content.end(), 
                                     [](unsigned char c) { 
                                         return c < 32 && c != 9 && c != 10 && c != 13; // 保留tab, LF, CR
                                     }), clean_content.end());
    
    // 3. 处理反斜杠（避免转义问题）
    size_t pos = 0;
    while ((pos = clean_content.find('\\', pos)) != std::string::npos) {
        // 将单个反斜杠替换为双反斜杠，或者直接移除
        clean_content.erase(pos, 1); // 简单移除反斜杠
    }
    
    // 4. 清理前后空格
    clean_content.erase(0, clean_content.find_first_not_of(" \t\n\r\f\v"));
    clean_content.erase(clean_content.find_last_not_of(" \t\n\r\f\v") + 1);
    
    // 5. 确保内容不为空
    if (clean_content.empty()) {
        clean_content = "true"; // 默认内容
    }
    
    // 6. 处理特殊的TPTP关键字冲突
    if (clean_content == "$false" || clean_content == "$true") {
        clean_content = "true"; // 避免与TPTP关键字冲突
    }
    
    // 检查是否已经有外层括号
    bool needs_parentheses = true;
    if (clean_content.length() >= 2 && 
        clean_content.front() == '(' && clean_content.back() == ')') {
        needs_parentheses = false;
    }
    
    if (needs_parentheses) {
        return "cnf(" + id + ", " + role + ", (" + clean_content + ")).";
    } else {
        return "cnf(" + id + ", " + role + ", " + clean_content + ").";
    }
}

// 引理处理相关工具函数实现

//** 计算引理的推导深度
//** @param lemmas 引理列表（会被修改，添加深度信息）
void calculateDerivationDepths(std::vector<Lemma>& lemmas) {
    // 构建从子句编号到引理的映射
    std::map<int, Lemma*> clause_map;
    for (auto& lemma : lemmas) {
        if (lemma.clause_number > 0) {
            clause_map[lemma.clause_number] = &lemma;
        }
    }
    
    // 检查是否包含CSE引理，如果是，需要特殊处理
    bool has_cse_lemmas = false;
    int min_cse_clause_number = INT_MAX;
    
    // 调试：统计引理类型和父子句信息
    int cse_lemma_count = 0;
    int cse_with_parents_count = 0;
    
    for (const auto& lemma : lemmas) {
        if (lemma.derivation_type == "cse") {
            cse_lemma_count++;
            has_cse_lemmas = true;
            if (lemma.clause_number > 0) {
                min_cse_clause_number = std::min(min_cse_clause_number, lemma.clause_number);
            }
            
            if (!lemma.parent_clause_numbers.empty()) {
                cse_with_parents_count++;
                // 显示前几个CSE引理的详细信息
                if (cse_with_parents_count <= 3) {
                }
            }
        }
    }
    

    
    // 初始化深度
    std::queue<Lemma*> bfs_queue;
    
    if (has_cse_lemmas) {

        
        // 对于CSE引理：直接初始化，在BFS过程中处理虚拟父子句
        
        // 初始化所有引理的深度
        for (auto& lemma : lemmas) {
            if (lemma.derivation_type == "cse") {
                // CSE引理初始深度为-1，等待计算
                lemma.derivation_depth = -1;
            } else {
                // 非CSE引理的原有逻辑
                if (lemma.parent_clause_numbers.empty() || 
                    lemma.source_info.find("input") != std::string::npos) {
                    lemma.derivation_depth = 0;
                    bfs_queue.push(&lemma);
                }
            }
        }
        
        // 对于CSE引理，立即计算那些所有父子句都是虚拟input的引理
        if (has_cse_lemmas) {
            bool found_computable = true;
            while (found_computable) {
                found_computable = false;
                for (auto& lemma : lemmas) {
                    if (lemma.derivation_type == "cse" && lemma.derivation_depth == -1) {
                        if (!lemma.parent_clause_numbers.empty()) {
                            bool all_parents_computed = true;
                            int max_parent_depth = 0;
                            
                            for (int parent_num : lemma.parent_clause_numbers) {
                                auto it = clause_map.find(parent_num);
                                if (it != clause_map.end() && it->second->derivation_depth >= 0) {
                                    // 找到了真实的父子句且已计算深度
                                    max_parent_depth = std::max(max_parent_depth, it->second->derivation_depth);
                                } else if (parent_num < min_cse_clause_number) {
                                    // 虚拟input子句，深度为0
                                    max_parent_depth = std::max(max_parent_depth, 0);
                                } else {
                                    // 父子句还未计算深度
                                    all_parents_computed = false;
                                    break;
                                }
                            }
                            
                            if (all_parents_computed) {
                                lemma.derivation_depth = max_parent_depth + 1;
                                bfs_queue.push(&lemma);
                                found_computable = true;
                            }
                        }
                    }
                }
            }
        }
    } else {
        // 原有的初始化逻辑：所有input子句（没有父子句的）深度为0
        for (auto& lemma : lemmas) {
            if (lemma.parent_clause_numbers.empty() || 
                lemma.source_info.find("input") != std::string::npos) {
                lemma.derivation_depth = 0;
                bfs_queue.push(&lemma);
            }
        }
    }
    
    // BFS计算推导深度
    while (!bfs_queue.empty()) {
        // 检查提前停止标志
        if (proof_found.load()) {
            break;
        }
        
        Lemma* current = bfs_queue.front();
        bfs_queue.pop();
        
        // 查找所有以current为父子句的引理
        for (auto& lemma : lemmas) {
            if (lemma.derivation_depth == -1) { // 未计算深度
                // 检查current是否是lemma的父子句
                bool is_child = false;
                for (int parent_num : lemma.parent_clause_numbers) {
                    if (parent_num == current->clause_number) {
                        is_child = true;
                        break;
                    }
                }
                
                if (is_child) {
                    // 检查所有父子句是否都已计算深度
                    bool all_parents_computed = true;
                    int max_parent_depth = 0;
                    
                    for (int parent_num : lemma.parent_clause_numbers) {
                        auto it = clause_map.find(parent_num);
                        if (it != clause_map.end()) {
                            // 找到了真实的父子句
                            if (it->second->derivation_depth == -1) {
                                all_parents_computed = false;
                                break;
                            } else {
                                max_parent_depth = std::max(max_parent_depth, it->second->derivation_depth);
                            }
                        } else if (has_cse_lemmas && parent_num < min_cse_clause_number) {
                            // 对于CSE引理，如果父子句编号小于最小CSE编号，直接视为深度0的input子句
                            max_parent_depth = std::max(max_parent_depth, 0);
                        } else {
                            // 找不到父子句，可能是引用了外部子句，暂时跳过
                            all_parents_computed = false;
                            break;
                        }
                    }
                    
                    if (all_parents_computed) {
                        lemma.derivation_depth = max_parent_depth + 1;
                        bfs_queue.push(&lemma);
                    }
                }
            }
        }
    }
    
    // 处理循环依赖：对于仍未计算深度的引理，尝试打破循环依赖
    std::vector<Lemma*> uncomputed_after_bfs;
    for (auto& lemma : lemmas) {
        if (lemma.derivation_depth == -1) {
            uncomputed_after_bfs.push_back(&lemma);
        }
    }
    
    // 迭代处理循环依赖，直到没有更多可以解决的循环
    int cycle_breaking_round = 1;
    while (!uncomputed_after_bfs.empty()) {
        
        // 分析循环依赖的根源
        std::map<int, int> problematic_parents;
        for (const auto* lemma : uncomputed_after_bfs) {
            for (int parent_num : lemma->parent_clause_numbers) {
                auto it = clause_map.find(parent_num);
                if (it != clause_map.end() && it->second->derivation_depth == -1) {
                    problematic_parents[parent_num]++;
                }
            }
        }
        
        // 如果没有找到问题父子句，说明可能是其他原因（如缺失父子句）
        if (problematic_parents.empty()) {
            
            // 尝试处理缺失父子句的情况：给这些引理分配一个合理的深度
            for (auto* lemma : uncomputed_after_bfs) {
                if (lemma->derivation_depth == -1) {
                    // 分析引理的父子句情况
                    int max_known_depth = 0;
                    bool has_known_parents = false;
                    
                    for (int parent_num : lemma->parent_clause_numbers) {
                        auto it = clause_map.find(parent_num);
                        if (it != clause_map.end() && it->second->derivation_depth >= 0) {
                            max_known_depth = std::max(max_known_depth, it->second->derivation_depth);
                            has_known_parents = true;
                        } else if (has_cse_lemmas && parent_num < min_cse_clause_number) {
                            max_known_depth = std::max(max_known_depth, 0);
                            has_known_parents = true;
                        }
                    }
                    
                    // 基于已知父子句或默认值设置深度
                    if (has_known_parents) {
                        lemma->derivation_depth = max_known_depth + 1;
                    } else {
                        // 如果没有任何已知父子句，设置为较深的层次
                        lemma->derivation_depth = cycle_breaking_round + 2;
                    }
                    
                }
            }
            break; // 退出循环依赖处理
        }
        
        // 找出最常见的问题父子句
        int most_problematic_parent = -1;
        int max_count = 0;
        for (const auto& pair : problematic_parents) {
            if (pair.second > max_count) {
                max_count = pair.second;
                most_problematic_parent = pair.first;
            }
        }
        
        if (most_problematic_parent != -1) {
            
            // 尝试通过设置一个合理的深度来打破循环
            auto it = clause_map.find(most_problematic_parent);
            if (it != clause_map.end()) {
                // 分析这个问题子句的父子句，尝试推断一个合理的深度
                int estimated_depth = cycle_breaking_round; // 基于轮次的默认深度
                
                // 如果它有父子句，尝试根据已知父子句推断深度
                if (!it->second->parent_clause_numbers.empty()) {
                    int known_parent_max_depth = -1;
                    for (int parent_num : it->second->parent_clause_numbers) {
                        auto parent_it = clause_map.find(parent_num);
                        if (parent_it != clause_map.end() && parent_it->second->derivation_depth >= 0) {
                            known_parent_max_depth = std::max(known_parent_max_depth, parent_it->second->derivation_depth);
                        } else if (has_cse_lemmas && parent_num < min_cse_clause_number) {
                            known_parent_max_depth = std::max(known_parent_max_depth, 0);
                        }
                    }
                    
                    if (known_parent_max_depth >= 0) {
                        estimated_depth = known_parent_max_depth + 1;
                    }
                }
                
                it->second->derivation_depth = estimated_depth;
                bfs_queue.push(it->second);
                
                // 继续BFS处理剩余的引理
                while (!bfs_queue.empty()) {
                    Lemma* current = bfs_queue.front();
                    bfs_queue.pop();
                    
                    for (auto& lemma : lemmas) {
                        if (lemma.derivation_depth == -1) {
                            bool is_child = false;
                            for (int parent_num : lemma.parent_clause_numbers) {
                                if (parent_num == current->clause_number) {
                                    is_child = true;
                                    break;
                                }
                            }
                            
                            if (is_child) {
                                bool all_parents_computed = true;
                                int max_parent_depth = 0;
                                
                                for (int parent_num : lemma.parent_clause_numbers) {
                                    auto parent_it = clause_map.find(parent_num);
                                    if (parent_it != clause_map.end()) {
                                        if (parent_it->second->derivation_depth == -1) {
                                            all_parents_computed = false;
                                            break;
                                        } else {
                                            max_parent_depth = std::max(max_parent_depth, parent_it->second->derivation_depth);
                                        }
                                    } else if (has_cse_lemmas && parent_num < min_cse_clause_number) {
                                        max_parent_depth = std::max(max_parent_depth, 0);
                                    } else {
                                        all_parents_computed = false;
                                        break;
                                    }
                                }
                                
                                if (all_parents_computed) {
                                    lemma.derivation_depth = max_parent_depth + 1;
                                    bfs_queue.push(&lemma);
                                }
                            }
                        }
                    }
                }
                
                // 重新统计未计算深度的引理
                std::vector<Lemma*> remaining_uncomputed;
                for (auto& lemma : lemmas) {
                    if (lemma.derivation_depth == -1) {
                        remaining_uncomputed.push_back(&lemma);
                    }
                }
                
                int resolved_in_this_round = uncomputed_after_bfs.size() - remaining_uncomputed.size();
                
                // 更新未计算列表，准备下一轮
                uncomputed_after_bfs = remaining_uncomputed;
                cycle_breaking_round++;
                
                // 防止无限循环
                if (cycle_breaking_round > 50) {
                    break;
                }
                
                // 如果这一轮没有解决任何引理，说明遇到了更复杂的循环
                if (resolved_in_this_round == 0) {
                    break;
                }
            } else {
                break;
            }
        } else {
            break;
        }
    }
    
    if (cycle_breaking_round > 1) {
    }
    
    // 对于CSE引理，清除临时的input标记，恢复为derived
    if (has_cse_lemmas) {
        for (auto& lemma : lemmas) {
            if (lemma.derivation_type == "cse" && lemma.source_info == "input") {
                lemma.source_info = "derived"; // 恢复为derived类型
            }
        }
    }
    
    // 统计推导深度分布
    std::map<int, int> depth_distribution;
    std::vector<const Lemma*> uncomputed_lemmas;
    
    for (const auto& lemma : lemmas) {
        if (lemma.derivation_depth >= 0) {
            depth_distribution[lemma.derivation_depth]++;
        } else {
            uncomputed_lemmas.push_back(&lemma);
        }
    }
    
    if (!uncomputed_lemmas.empty()) {
        
        // 限制输出数量，避免过多信息
        int max_display = std::min(10, static_cast<int>(uncomputed_lemmas.size()));
        for (int i = 0; i < max_display; ++i) {
            const Lemma* lemma = uncomputed_lemmas[i];
            
            // 分析为什么未计算深度
            if (!lemma->parent_clause_numbers.empty()) {
                std::vector<int> missing_parents;
                std::vector<int> uncomputed_parents;
                bool has_virtual_parents = false;
                
                for (int parent_num : lemma->parent_clause_numbers) {
                    auto it = clause_map.find(parent_num);
                    if (it == clause_map.end()) {
                        if (has_cse_lemmas && parent_num < min_cse_clause_number) {
                            has_virtual_parents = true;
                        } else {
                            missing_parents.push_back(parent_num);
                        }
                    } else if (it->second->derivation_depth == -1) {
                        uncomputed_parents.push_back(parent_num);
                    }
                }
                // 消除未使用变量警告
                (void)has_virtual_parents;
                (void)missing_parents;
                (void)uncomputed_parents;
            }
        }
        
        // 统计未计算深度引理的类型分布（但不输出）
        std::map<std::string, int> uncomputed_by_type;
        std::map<std::string, int> uncomputed_by_source;
        for (const auto* lemma : uncomputed_lemmas) {
            uncomputed_by_type[lemma->derivation_type]++;
            uncomputed_by_source[lemma->source_info]++;
        }
    }
}

//** 单元子句识别函数（包含等式）
//** 单元子句是只包含一个文字的子句
//** @param content 引理内容字符串
//** @return 如果是单元子句返回true，否则返回false
bool is_unit_clause(const std::string& content) {
    if (content.empty()) return false;
    
    // 移除前后空格
    std::string trimmed = content;
    trimmed.erase(0, trimmed.find_first_not_of(" \t\n\r\f\v"));
    trimmed.erase(trimmed.find_last_not_of(" \t\n\r\f\v") + 1);
    
    // 长度预筛选
    if (trimmed.length() > 100) {
        return false;
    }
    
    // 单元子句不能包含析取符号 |
    if (trimmed.find('|') != std::string::npos) {
        return false;
    }

    // 模式1：正原子谓词 P 或 P(args)
    std::regex positive_atom(R"(^\s*[a-zA-Z_][a-zA-Z0-9_]*\s*(\([^|]*\))?\s*$)");
    if (std::regex_match(trimmed, positive_atom)) {
        return true;
    }
    
    // 模式2：否定原子 ~P 或 ~P(args)
    std::regex negative_atom(R"(^\s*~\s*[a-zA-Z_][a-zA-Z0-9_]*\s*(\([^|]*\))?\s*$)");
    if (std::regex_match(trimmed, negative_atom)) {
        return true;
    }
    
    // 模式3：等式文字 term1 = term2
    std::regex equality_literal(R"(^\s*[^|=]+\s*=\s*[^|=]+\s*$)");
    if (std::regex_match(trimmed, equality_literal)) {
        // 确保等号两边不包含逻辑连接符
        size_t eq_pos = trimmed.find('=');
        if (eq_pos != std::string::npos) {
            std::string left_side = trimmed.substr(0, eq_pos);
            std::string right_side = trimmed.substr(eq_pos + 1);
            
            // 检查两边是否包含析取符号（CNF中不会有合取符号）
            if (left_side.find('|') == std::string::npos && 
                right_side.find('|') == std::string::npos) {
                return true;
            }
        }
    }
    
    // 模式4：否定等式 ~(term1 = term2)
    std::regex negated_equality(R"(^\s*~\s*\(\s*[^|=()]+\s*=\s*[^|=()]+\s*\)\s*$)");
    if (std::regex_match(trimmed, negated_equality)) {
        return true;
    }
    
    // 模式5：不等式文字 term1 != term2
    std::regex inequality_literal(R"(^\s*[^|!=]+\s*!=\s*[^|!=]+\s*$)");
    if (std::regex_match(trimmed, inequality_literal)) {
        return true;
    }
    
    // 如果都不匹配，则不是标准的单元子句
    return false;
}

//** 过滤和选择引理
//** 根据引理的长度、内容、来源信息和AVATAR约束进行过滤
//** 同时提供详细的统计信息，帮助分析引理质量和分布
//** @param all_lemmas 所有提取的引理列表
//** @param unit_clauses_only 是否只保留单元子句
//** @return 过滤后的引理列表
std::vector<Lemma> filter_lemmas(const std::vector<Lemma>& all_lemmas, bool unit_clauses_only) {
    std::vector<Lemma> filtered;
    const size_t max_length = 200; // 调整字符长度限制

    // 统计信息
    std::map<std::string, int> type_count;
    std::map<std::string, int> avatar_count;
    std::map<std::string, int> source_count;
    int total_input = 0, total_resolution = 0, total_other = 0;
    int avatar_filtered_count = 0; // 被AVATAR过滤掉的数量
    int unit_clause_count = 0;     // 单元子句数量
    int non_unit_filtered_count = 0; // 被单元子句过滤掉的数量

    for (const auto& lemma : all_lemmas) {
        // 基本统计
        type_count[lemma.derivation_type]++;
        if (!lemma.avatar_id.empty()) {
            avatar_count["有AVATAR约束"]++;
        } else {
            avatar_count["无AVATAR约束"]++;
        }
        
        // 来源统计
        if (lemma.source_info.find("input") != std::string::npos) {
            total_input++;
            source_count["input"]++;
        } else if (lemma.source_info.find("resolution") != std::string::npos) {
            total_resolution++;
            source_count["resolution"]++;
        } else if (lemma.source_info.find("superposition") != std::string::npos) {
            source_count["superposition"]++;
            total_other++;
        } else if (lemma.source_info.find("forward demodulation") != std::string::npos) {
            source_count["forward_demodulation"]++;
            total_other++;
        } else {
            source_count["其他"]++;
            total_other++;
        }

        // 检查是否为单元子句
        bool is_unit = is_unit_clause(lemma.content);
        if (is_unit) {
            unit_clause_count++;
        }

        // 过滤策略
        bool should_include = true;
        
        // 1. 基本长度和内容过滤
        if (lemma.content.length() >= max_length ||
            lemma.content.find("$false") != std::string::npos ||
            lemma.content.find("$true") != std::string::npos ||
            lemma.content.empty()) {
            should_include = false;
        }
        
        // 2. 基于来源信息的过滤
        if (should_include) {
            // 排除input子句
            if (lemma.source_info.find("input") != std::string::npos) {
                should_include = false;
            }
            // 排除avatar相关子句
            else if (lemma.source_info.find("avatar") != std::string::npos) {
                should_include = false;
            }
        }
        
        // 3. AVATAR约束过滤 - 始终过滤掉所有有AVATAR约束的引理
        if (should_include) {
            bool has_avatar_constraint = false;
            
            // 检查avatar_id字段
            if (!lemma.avatar_id.empty()) {
                has_avatar_constraint = true;
            }
            
            // 检查content中是否包含AVATAR约束模式 "<-"
            if (lemma.content.find("<-") != std::string::npos) {
                has_avatar_constraint = true;
            }
            
            if (has_avatar_constraint) {
                should_include = false;
                avatar_filtered_count++;
            }
        }
        
        // 4. 单元子句过滤策略
        if (should_include && unit_clauses_only) {
            if (!is_unit) {
                should_include = false;
                non_unit_filtered_count++;
            }
        }
        
        if (should_include) {
             filtered.push_back(lemma);
        }
    }
    
    // 输出详细统计信息

    // 统计信息已收集，但不输出
    
    return filtered;
}

//** 创建新的CNF问题文件，包含原始问题和指定的引理集
bool create_new_cnf_problem_files(const std::string& cnf_problem_path,
                                const std::vector<std::vector<Lemma>>& lemma_sets,
                                const std::string& output_file_prefix) {
    std::ifstream cnf_file(cnf_problem_path);
    if (!cnf_file.is_open()) {
        return false;
    }
    
    // 读取原始CNF内容
    std::stringstream cnf_content_ss;
    cnf_content_ss << cnf_file.rdbuf();
    cnf_file.close();
    std::string cnf_content = cnf_content_ss.str();

    bool all_created = true;
    for (size_t i = 0; i < lemma_sets.size(); ++i) {
        std::string new_file_path = output_file_prefix + std::to_string(i + 1) + ".cnf";
        std::ofstream new_file(new_file_path);
        if (!new_file.is_open()) {
            all_created = false;
            continue;
        }

        // 写入原始CNF内容
        new_file << cnf_content;
        
        // 添加引理分隔注释
        new_file << "\n% ========================================\n";
        new_file << "% 自动添加的引理 (集合 " << i + 1 << ")\n";
        new_file << "% 总共 " << lemma_sets[i].size() << " 个引理\n";
        new_file << "% ========================================\n\n";
        
        for (const auto& lemma : lemma_sets[i]) {
            new_file << lemma.to_tptp_cnf() << "\n";
        }
        
        new_file.close();
    }
    return all_created;
}

//** 作为独立进程运行证明器，并支持提前停止
void run_prover_as_process(const std::string& prover_exec_path,
                          ProcessResult* result_ptr,
                          int proof_time_limit,
                          const std::string& proof_mode_options,
                          int memory_limit_mb,
                          bool enable_early_termination,
                          bool enable_cpu_affinity,
                          const std::string& prover_type) {
    if (!result_ptr) {
        return;
    }
    
    auto start_time = std::chrono::high_resolution_clock::now();
    


    if (!file_exists(result_ptr->problem_file)) {
        return;
    }

    // 构建命令参数
    std::vector<std::string> args;
    args.push_back(prover_exec_path);
    
    // 解析 proof_mode_options
    std::istringstream iss(proof_mode_options);
    std::string token;
    while (iss >> token) {
        args.push_back(token);
    }
    
    // 根据证明器类型添加时间和内存限制参数
    if (prover_type == "eprover") {
        // E-Prover参数格式
        // 使用--soft-cpu-limit确保在壁钟时间内结束，--cpu-limit作为硬限制
        args.push_back("--soft-cpu-limit=" + std::to_string(proof_time_limit));
        args.push_back("--cpu-limit=" + std::to_string(proof_time_limit + 30)); // 硬限制稍长一些
        
        if (memory_limit_mb > 0) {
            args.push_back("--memory-limit=" + std::to_string(memory_limit_mb));
        }
    } else {
        // Vampire参数格式（默认）
        args.push_back("-t");
        args.push_back(std::to_string(proof_time_limit));
        
        if (memory_limit_mb > 0) {
            args.push_back("--memory_limit");
            args.push_back(std::to_string(memory_limit_mb));
        }
    }
    
    args.push_back(result_ptr->problem_file);

    // 构建命令字符串用于日志
    std::string command_str = prover_exec_path;
    for (size_t i = 1; i < args.size(); ++i) {
        command_str += " " + args[i];
    }
    command_str += " > \"" + result_ptr->result_file + "\" 2>&1";

    // 使用fork/exec启动进程
    pid_t child_pid = fork();
    if (child_pid == -1) {
        return;
    }
    
    if (child_pid == 0) {
        // 子进程：设置CPU亲和性（如果启用）
        if (enable_cpu_affinity) {
            cpu_set_t cpuset;
            CPU_ZERO(&cpuset);
            
            // 获取系统CPU数量
            int num_cpus = sysconf(_SC_NPROCESSORS_ONLN);
            
            // 计算目标CPU核心
            int target_cpu;
            if (result_ptr->process_id == 0) {
                // 基准进程使用最后一个核心，避免与引理增强进程竞争
                target_cpu = num_cpus - 1;
            } else {
                // 引理增强进程从核心0开始依次分配
                target_cpu = (result_ptr->process_id - 1) % (num_cpus - 1);
            if (target_cpu < 0) target_cpu = 0; // 防止负数
            }
            
            CPU_SET(target_cpu, &cpuset);
            
            sched_setaffinity(0, sizeof(cpuset), &cpuset);
        }
        
        // 重定向输出并执行vampire
        int fd = open(result_ptr->result_file.c_str(), O_WRONLY | O_CREAT | O_TRUNC, 0644);
        if (fd != -1) {
            dup2(fd, STDOUT_FILENO);
            dup2(fd, STDERR_FILENO);
            close(fd);
        }
        
        // 准备execv参数
        std::vector<char*> argv_exec;
        for (const auto& arg : args) {
            argv_exec.push_back(const_cast<char*>(arg.c_str()));
        }
        argv_exec.push_back(nullptr);
        
        execv(prover_exec_path.c_str(), argv_exec.data());
        
        // 如果execv失败
        exit(1);
    } else {
        // 父进程：保存子进程PID并等待完成
        result_ptr->vampire_process_id = child_pid;
        
        int status;
        bool process_terminated = false;
        
        // 定期检查提前停止标志和进程状态
        while (!process_terminated) {
            // 检查提前停止条件
            if (enable_early_termination && proof_found.load() && 
                proof_finder_id.load() != result_ptr->process_id) {
                
                // 使用互斥锁保护输出，避免多进程输出混乱
                {
                    std::lock_guard<std::mutex> output_lock(early_termination_mutex);
                }
                
                // 发送SIGTERM信号
                kill(child_pid, SIGTERM);
                
                // 缩短等待时间，快速响应全局超时
                for (int i = 0; i < 3; ++i) {
                    if (waitpid(child_pid, &status, WNOHANG) == child_pid) {
                        process_terminated = true;
                        break;
                    }
                    usleep(100000); // 100ms，总共最多300ms
                }
                
                // 如果还没退出，强制终止
                if (!process_terminated) {
                    std::lock_guard<std::mutex> output_lock(early_termination_mutex);
                    kill(child_pid, SIGKILL);
                    waitpid(child_pid, &status, 0);
                    process_terminated = true;
                }
                
                result_ptr->exit_code = -2; // 特殊标记：提前停止
                break;
            }
            
            // 检查进程是否自然结束
            if (waitpid(child_pid, &status, WNOHANG) == child_pid) {
                process_terminated = true;
                result_ptr->exit_code = WEXITSTATUS(status);
                break;
            }
            
            // 短暂休眠后继续检查
            usleep(50000); // 50ms
        }
        
        // 计算执行时间
        auto end_time = std::chrono::high_resolution_clock::now();
        auto duration = std::chrono::duration_cast<std::chrono::milliseconds>(end_time - start_time);
        result_ptr->execution_time_seconds = duration.count() / 1000.0;
        
        // 解析结果文件获取SZS状态（与原始文件逻辑一致）
        std::ifstream result_file(result_ptr->result_file);
        std::string line;
        result_ptr->szs_status = "$SZS Status Unknown"; // 默认状态

        if (result_file.is_open()) {
            std::regex szs_status_regex1(R"(^\s*%\s*SZS\s+status\s+(\S+)\s+for)"); 
            std::regex szs_status_regex2(R"(^\s*\$SZS\s+Status\s+(\S+)\s*$)"); 
            std::regex szs_status_regex3(R"(^\s*SZS\s+Status\s+(\S+)\s*$)");
            std::regex szs_status_regex4(R"(^\s*#\s*SZS\s+status\s+(\S+)\s*$)"); // E-Prover格式

            while (std::getline(result_file, line)) {
                std::smatch match;
                
                // 尝试匹配第一种格式
                if (std::regex_search(line, match, szs_status_regex1)) {
                    if (match.size() > 1) {
                        result_ptr->szs_status = "$SZS Status " + match[1].str();
                        break;
                    }
                }
                // 尝试匹配第二种格式
                else if (std::regex_search(line, match, szs_status_regex2)) {
                    if (match.size() > 1) {
                        result_ptr->szs_status = "$SZS Status " + match[1].str();
                        break;
                    }
                }
                // 尝试匹配第三种格式
                else if (std::regex_search(line, match, szs_status_regex3)) {
                    if (match.size() > 1) {
                        result_ptr->szs_status = "$SZS Status " + match[1].str();
                        break;
                    }
                }
                // 尝试匹配第四种格式（E-Prover）
                else if (std::regex_search(line, match, szs_status_regex4)) {
                    if (match.size() > 1) {
                        result_ptr->szs_status = "$SZS Status " + match[1].str();
                        break;
                    }
                }
            }
            result_file.close();
        } else {
            result_ptr->szs_status = "$SZS Status Error: Result file not accessible.";
        }
        
        // 检查是否找到证明
        result_ptr->found_proof = (result_ptr->szs_status.find("Theorem") != std::string::npos || 
                                  result_ptr->szs_status.find("Unsatisfiable") != std::string::npos ||
                                  result_ptr->szs_status.find("Satisfiable") != std::string::npos);

        // 使用互斥锁保护输出，避免多进程输出混乱
        {
            std::lock_guard<std::mutex> output_lock(early_termination_mutex);
        }

        if (result_ptr->found_proof) {
            // 如果启用提前停止，使用互斥锁保护提前停止逻辑和输出
            if (enable_early_termination) {
                std::lock_guard<std::mutex> lock(early_termination_mutex);
                
                // 检查是否已经有其他进程触发了提前停止
                if (!proof_found.load()) {
                    proof_found.store(true);
                    proof_finder_id.store(result_ptr->process_id);
                    
                    // 根据结果类型输出不同信息
                    if (result_ptr->szs_status.find("Theorem") != std::string::npos || 
                        result_ptr->szs_status.find("Unsatisfiable") != std::string::npos) {
                        // 找到证明
                    } else if (result_ptr->szs_status.find("Satisfiable") != std::string::npos) {
                        // 确定问题是可满足的（不是定理）
                    }
                } else {
                }
            } else {
                // 没有启用提前停止时，仍然需要保护输出
                std::lock_guard<std::mutex> output_lock(early_termination_mutex);
            }
        }
    }
}

//** 根据CNF文件大小自适应计算时间限制
int calculate_adaptive_time_limit(int cnf_clause_count, int user_specified_time) {
    // 如果明确指定了时间，则优先使用设置的时间 
    if (user_specified_time > 0) {
        return user_specified_time;
    }
    
    if (cnf_clause_count <= 0) {
        return 5; // 默认5秒
    }
    
    if (cnf_clause_count <= 10000) {
        // 小文件：使用默认5秒
        return 5;
    } else {
        // 大文件：每1万子句增加1秒
        double additional_time = (cnf_clause_count - 10000) / 10000.0;
        int adjusted_time = 5 + static_cast<int>(std::round(additional_time));
        
        // 设置合理上限，避免时间过长
        return (adjusted_time < 300) ? adjusted_time : 300;
    }
}

//** 深度相关的引理过滤函数
//** @param lemmas_with_depths 带深度信息的引理列表
//** @param unit_clauses_only 是否只保留单元子句
//** @return 过滤后的引理列表
std::vector<Lemma> filter_lemmas_for_depth_based(const std::vector<Lemma>& lemmas_with_depths, bool unit_clauses_only) {
    std::vector<Lemma> filtered;
    const size_t max_length = 200;
    
    int input_filtered = 0;
    int depth_negative_filtered = 0;
    int unit_filtered = 0;
    int content_filtered = 0;
    int avatar_filtered = 0;
    
    // 存储被过滤的引理样本（前5个）
    std::vector<std::pair<std::string, std::string>> filtered_samples; // reason, content
    
    for (const auto& lemma : lemmas_with_depths) {
        bool should_include = true;
        std::string filter_reason;
        
        // 1. 过滤input子句和深度0的引理
        if (lemma.source_info.find("input") != std::string::npos || 
            lemma.derivation_depth == 0) {
            should_include = false;
            input_filtered++;
            filter_reason = "input子句或深度0";
        }
        
        // 2. 过滤深度为负的引理
        else if (lemma.derivation_depth < 0) {
            should_include = false;
            depth_negative_filtered++;
            filter_reason = "深度计算失败(负值)";
        }
        
        // 3. 基本内容过滤
        else if (lemma.content.length() >= max_length ||
                 lemma.content.find("$false") != std::string::npos ||
                 lemma.content.find("$true") != std::string::npos ||
                 lemma.content.empty()) {
            should_include = false;
            content_filtered++;
            if (lemma.content.length() >= max_length) {
                filter_reason = "内容过长(>=" + std::to_string(max_length) + "字符)";
            } else if (lemma.content.find("$false") != std::string::npos) {
                filter_reason = "包含$false";
            } else if (lemma.content.find("$true") != std::string::npos) {
                filter_reason = "包含$true";
            } else {
                filter_reason = "内容为空";
            }
        }
        
        // 4. avatar约束过滤
        else if (!lemma.avatar_id.empty()) {
            should_include = false;
            avatar_filtered++;
            filter_reason = "Avatar约束(avatar_id非空)";
        }
        
        // 5. 单元子句过滤
        else if (unit_clauses_only && !is_unit_clause(lemma.content)) {
            should_include = false;
            unit_filtered++;
            filter_reason = "非单元子句";
        }
        
        if (!should_include && filtered_samples.size() < 5) {
            // 记录前5个被过滤的引理样本
            std::string sample_content = lemma.content;
            if (sample_content.length() > 80) {
                sample_content = sample_content.substr(0, 80) + "...";
            }
            filtered_samples.push_back({filter_reason, sample_content});
        }
        
        if (should_include) {
            filtered.push_back(lemma);
        }
    }
    
    // 统计信息已收集，但不输出
    
    return filtered;
}

int merge_cse_lemmas_to_cnf(const std::string& base_cnf_path,
                           const std::string& cse_output_dir,
                           const std::string& merged_cnf_path) {
    // 合并引理处理，但不输出详细信息
    
    // 打开基础CNF文件
    std::ifstream base_file(base_cnf_path);
    if (!base_file.is_open()) {
        return -1;
    }
    
    // 创建合并后的文件
    std::ofstream merged_file(merged_cnf_path);
    if (!merged_file.is_open()) {
        base_file.close();
        return -1;
    }
    
    // 复制基础CNF文件内容
    merged_file << "% Merged CNF file with CSE lemmas\n";
    merged_file << "% Base CNF file: " << base_cnf_path << "\n";
    merged_file << "% CSE output directory: " << cse_output_dir << "\n\n";
    
    std::string line;
    int base_cnf_count = 0;
    while (std::getline(base_file, line)) {
        merged_file << line << "\n";
        if (line.find("cnf(") != std::string::npos) {
            base_cnf_count++;
        }
    }
    base_file.close();
    

    
    // 查找CSE输出目录中的.out文件
    std::string find_command = "find " + cse_output_dir + " -name \"*.out\" 2>/dev/null";
    FILE* pipe = popen(find_command.c_str(), "r");
    if (!pipe) {
        merged_file.close();
        return 0; // 返回0表示没有找到引理，但不是错误
    }
    
    char buffer[1024];
    std::vector<std::string> out_files;
    while (fgets(buffer, sizeof(buffer), pipe) != nullptr) {
        std::string filepath(buffer);
        // 移除换行符
        if (!filepath.empty() && filepath.back() == '\n') {
            filepath.pop_back();
        }
        out_files.push_back(filepath);
    }
    pclose(pipe);
    
    if (out_files.empty()) {
        merged_file.close();
        return 0;
    }
    

    
    // 处理每个.out文件
    int total_lemma_count = 0;
    int lemma_id_counter = 1;
    
    merged_file << "\n% === CSE Generated Lemmas ===\n";
    
    for (const auto& out_file : out_files) {
        
        std::ifstream lemma_file(out_file);
        if (!lemma_file.is_open()) {
            continue;
        }
        
        std::string lemma_line;
        int file_lemma_count = 0;
        
        while (std::getline(lemma_file, lemma_line)) {
            // 跳过空行和注释行
            if (lemma_line.empty() || lemma_line[0] == '%' || lemma_line[0] == '#') {
                continue;
            }
            
            // 检查是否已经是CNF格式
            if (lemma_line.find("cnf(") != std::string::npos) {
                // 已经是CNF格式，直接添加
                merged_file << lemma_line << "\n";
                file_lemma_count++;
            } else {
                // 尝试将引理转换为CNF格式
                // 这里做简单的格式化处理
                std::string cnf_lemma = "cnf(cse_lemma_" + std::to_string(lemma_id_counter) + 
                                       ", axiom, (" + lemma_line + ")).";
                merged_file << cnf_lemma << "\n";
                file_lemma_count++;
                lemma_id_counter++;
            }
        }
        
        lemma_file.close();
        total_lemma_count += file_lemma_count;
    }
    
    merged_file << "\n% === End of CSE Lemmas ===\n";
    merged_file.close();
    
    return total_lemma_count;
}

// --- LemmaFormatConverter 实现 ---

Lemma LemmaFormatConverter::convertCSECnfToLemma(const std::string& cse_cnf_line, int& lemma_counter) {
    (void)lemma_counter; // 消除未使用参数警告，CSE引理保持原始ID
    Lemma lemma;
    
    // 基本格式检查
    if (cse_cnf_line.empty() || cse_cnf_line.find("cnf(") != 0) {
        return lemma; // 返回空引理
    }
    
    try {
        // 解析CSE cnf格式，更灵活的解析方式
        // 格式：cnf(id, role, (content), inference(...)).
        
        // 首先找到各个关键位置
        size_t cnf_start = cse_cnf_line.find("cnf(");
        if (cnf_start == std::string::npos) {
            return lemma;
        }
        
        size_t content_start = cnf_start + 4; // 跳过"cnf("
        
        // 找到第一个逗号（ID后面）
        size_t first_comma = cse_cnf_line.find(',', content_start);
        if (first_comma == std::string::npos) {
            return lemma;
        }
        
        // 提取ID
        std::string cse_id = cse_cnf_line.substr(content_start, first_comma - content_start);
        cse_id.erase(0, cse_id.find_first_not_of(" \t\n\r\f\v"));
        cse_id.erase(cse_id.find_last_not_of(" \t\n\r\f\v") + 1);
        
        // 找到第二个逗号（role后面）
        size_t second_comma = cse_cnf_line.find(',', first_comma + 1);
        if (second_comma == std::string::npos) {
            return lemma;
        }
        
        // 提取role
        std::string role = cse_cnf_line.substr(first_comma + 1, second_comma - first_comma - 1);
        role.erase(0, role.find_first_not_of(" \t\n\r\f\v"));
        role.erase(role.find_last_not_of(" \t\n\r\f\v") + 1);
        
        // 找到content的开始位置（第一个括号）
        size_t content_paren_start = cse_cnf_line.find('(', second_comma);
        if (content_paren_start == std::string::npos) {
            return lemma;
        }
        
        // 找到content的结束位置（匹配的右括号）
        int paren_count = 1;
        size_t content_paren_end = content_paren_start + 1;
        while (content_paren_end < cse_cnf_line.length() && paren_count > 0) {
            if (cse_cnf_line[content_paren_end] == '(') {
                paren_count++;
            } else if (cse_cnf_line[content_paren_end] == ')') {
                paren_count--;
            }
            content_paren_end++;
        }
        
        if (paren_count != 0) {
            // 括号不匹配，尝试简单的查找方式
            content_paren_end = cse_cnf_line.find("), inference(", content_paren_start);
            if (content_paren_end == std::string::npos) {
                return lemma;
            }
            content_paren_end++; // 包含右括号
        } else {
            content_paren_end--; // 调整到右括号位置
        }
        
        // 查找inference信息
        size_t inference_start = cse_cnf_line.find("inference(", content_paren_end);
        std::string inference_str = "";
        if (inference_start != std::string::npos) {
            size_t inference_end = cse_cnf_line.find("))", inference_start);
            if (inference_end != std::string::npos) {
                inference_str = cse_cnf_line.substr(inference_start + 10, inference_end - inference_start - 10);
            }
        }
        
        // 提取纯子句内容（用于处理）
        std::string content = cse_cnf_line.substr(content_paren_start + 1, content_paren_end - content_paren_start - 1);
        content.erase(0, content.find_first_not_of(" \t\n\r\f\v"));
        content.erase(content.find_last_not_of(" \t\n\r\f\v") + 1);
        
        // 构建统一的引理对象
        lemma.id = cse_id; // 保持原始的CSE ID，如i_0_xxx
        lemma.role = role;
        // 保存纯子句内容用于引理处理
        lemma.content = content;
        // 保存完整的原始TPTP行用于输出
        lemma.original_tptp_line = cse_cnf_line;
        lemma.clause_number = extractCSEClauseNumber(cse_id);
        lemma.derivation_type = "cse"; // 标记为CSE生成
        
        // 解析推理信息
        std::string inference_rule;
        std::vector<int> parent_numbers;
        if (!inference_str.empty()) {
            parseCSEInferenceInfo(inference_str, inference_rule, parent_numbers);
        }
        
        lemma.inference_rule = inference_rule;
        lemma.parent_clause_numbers = parent_numbers;
        
        // CSE生成的引理都是推导引理，没有input类型
        lemma.source_info = "derived";
        
        lemma.avatar_id = "";
        lemma.derivation_depth = -1; // 稍后计算
        
        return lemma;
        
    } catch (const std::exception& e) {
        // 解析CSE cnf行时出错，跳过该行
    }
    
    return lemma; // 返回空引理表示解析失败
}

Lemma LemmaFormatConverter::convertVampireLemmaToUnified(const std::string& vampire_line, int& lemma_counter) {
 
    Lemma lemma;
    
    try {
        std::regex vampire_regex(R"(^\s*\[SA\]\s+(new|active|passive):\s*(\d+)\.\s*(.*?)(?:\s+\[([^\]]*)\])?\s*$)");
        std::smatch match;
        
        if (std::regex_search(vampire_line, match, vampire_regex)) {
            std::string derivation_type = match[1].str();
            std::string clause_num_str = match[2].str();
            std::string content = match[3].str();
            std::string inference_info = (match.size() > 4 && match[4].matched) ? match[4].str() : "";
            
            lemma.id = "vampireLemma" + std::to_string(lemma_counter++);
            lemma.role = "axiom";
            lemma.content = content;
            lemma.derivation_type = derivation_type;
            lemma.clause_number = std::stoi(clause_num_str);
            
            // 简化的推理信息解析
            if (inference_info.find("input") != std::string::npos) {
                lemma.source_info = "input";
            } else {
                lemma.source_info = "derived";
                lemma.inference_rule = "vampire_inference";
            }
            
            lemma.avatar_id = "";
            lemma.derivation_depth = -1;
        }
    } catch (const std::exception& e) {
        // 解析Vampire引理行时出错，跳过该行
    }
    
    return lemma;
}

Lemma LemmaFormatConverter::cleanLemmaForMerging(const Lemma& lemma) {
    Lemma cleaned = lemma;
    
    // 清理证明器特定的信息
    cleaned.parent_clause_numbers.clear(); // 移除具体的父子句编号
    cleaned.avatar_id = "";                // 移除avatar信息
    cleaned.derivation_depth = -1;         // 重置推导深度
    
    // 保留通用信息：id, role, content, source_info
    // 但简化inference_rule为通用类型
    if (!cleaned.inference_rule.empty()) {
        if (cleaned.inference_rule.find("resolution") != std::string::npos) {
            cleaned.inference_rule = "resolution";
        } else if (cleaned.inference_rule.find("superposition") != std::string::npos) {
            cleaned.inference_rule = "superposition";
        } else {
            cleaned.inference_rule = "other";
        }
    }
    
    return cleaned;
}

std::vector<Lemma> LemmaFormatConverter::parseCSEOutputDirectory(const std::string& cse_output_directory) {
    std::vector<Lemma> lemmas;
    int lemma_counter = 1;
    

    
    // 检查目录是否存在
    std::ifstream check_dir(cse_output_directory);
    if (!check_dir.good()) {
        // CSE输出目录不存在
        return lemmas;
    }
    
    // 提取问题名称（从目录路径中）
    std::string problem_name;
    size_t last_slash = cse_output_directory.find_last_of("/\\");
    if (last_slash != std::string::npos) {
        std::string dir_name = cse_output_directory.substr(last_slash + 1);
        // 目录格式：问题名+其他后缀，尝试提取问题名
        // 常见格式：问题名_short+后缀，但也可能有其他格式
        size_t suffix_pos = dir_name.find("_");
        if (suffix_pos != std::string::npos) {
            problem_name = dir_name.substr(0, suffix_pos);
        } else {
            problem_name = dir_name; // 如果没有下划线，整个目录名就是问题名
        }
    } else {
        problem_name = cse_output_directory; // 备用方案
    }
    
    // 查找目录中的.out文件
    std::string cse_output_file;
    
    std::string find_cmd = "find \"" + cse_output_directory + "\" -maxdepth 1 -name \"*.out\" 2>/dev/null | head -1";
    FILE* pipe = popen(find_cmd.c_str(), "r");
    if (pipe) {
        char buffer[1024];
        if (fgets(buffer, sizeof(buffer), pipe) != nullptr) {
            cse_output_file = std::string(buffer);
            // 移除换行符
            if (!cse_output_file.empty() && cse_output_file.back() == '\n') {
                cse_output_file.pop_back();
            }
        }
        pclose(pipe);
    }
    
    // 如果没找到，使用默认文件名
    if (cse_output_file.empty()) {
        cse_output_file = cse_output_directory + "/" + problem_name + "_short+eq_rstfp.out";
    }
    

    
    // 检查文件是否存在
    std::ifstream file(cse_output_file);
    if (!file.is_open()) {
        // 无法找到任何CSE输出文件
        return lemmas;
    }
    

    
    // 解析单一.out文件
    std::string line;
    int total_parsed = 0;
    while (std::getline(file, line)) {
        if (line.empty() || line[0] == '%') {
            continue; // 跳过空行和注释
        }
        
        Lemma lemma = convertCSECnfToLemma(line, lemma_counter);
        if (!lemma.id.empty()) { // 检查是否成功解析
            lemmas.push_back(lemma);
            total_parsed++;
        }
    }
    
    file.close();

    return lemmas;
}

int LemmaFormatConverter::extractCSEClauseNumber(const std::string& cse_id) {
    // 从CSE ID中提取数字，例如 i_0_13 -> 13
    std::regex number_regex(R"(_(\d+)$)");
    std::smatch match;
    
    if (std::regex_search(cse_id, match, number_regex)) {
        try {
            return std::stoi(match[1].str());
        } catch (...) {
            return -1;
        }
    }
    
    return -1;
}

void LemmaFormatConverter::parseCSEInferenceInfo(const std::string& inference_str, 
                                                std::string& rule, 
                                                std::vector<int>& parent_numbers) {
    rule = "";
    parent_numbers.clear();
    
    try {
        
        // 首先提取推理规则名称（第一个逗号前的部分）
        size_t first_comma = inference_str.find(',');
        if (first_comma != std::string::npos) {
            rule = inference_str.substr(0, first_comma);
            rule.erase(0, rule.find_first_not_of(" \t\n\r\f\v"));
            rule.erase(rule.find_last_not_of(" \t\n\r\f\v") + 1);
        }
        
        // 查找最后一个方括号对，这里包含父子句信息
        size_t last_bracket_start = inference_str.rfind('[');
        size_t last_bracket_end = inference_str.rfind(']');
        
        if (last_bracket_start != std::string::npos && last_bracket_end != std::string::npos && 
            last_bracket_start < last_bracket_end) {
            
            std::string parents_str = inference_str.substr(last_bracket_start + 1, 
                                                          last_bracket_end - last_bracket_start - 1);
            
            // 解析父子句列表
            if (!parents_str.empty()) {
                // 处理两种格式：
                // 1. i_0_11,i_0_10 (CSE格式)
                // 2. 1445 (简单数字格式)
                
                std::stringstream ss(parents_str);
                std::string item;
                
                while (std::getline(ss, item, ',')) {
                    item.erase(0, item.find_first_not_of(" \t\n\r\f\v"));
                    item.erase(item.find_last_not_of(" \t\n\r\f\v") + 1);
                    
                    if (item.empty()) continue;
                    
                    // 尝试提取数字
                    int parent_num = -1;
                    
                    // 格式1: i_0_数字
                    std::regex cse_id_regex(R"(i_\d+_(\d+))");
                    std::smatch match;
                    if (std::regex_search(item, match, cse_id_regex)) {
                        try {
                            parent_num = std::stoi(match[1].str());
                        } catch (...) {
                            continue;
                        }
                    } else {
                        // 格式2: 直接的数字
                        try {
                            parent_num = std::stoi(item);
                        } catch (...) {
                            continue;
                        }
                    }
                    
                    if (parent_num >= 0) {
                        parent_numbers.push_back(parent_num);
                    }
                }
            }
        }
        
    } catch (const std::exception& e) {
                        // 解析CSE推理信息时出错，跳过
    }
} 

// 孤儿进程检测和清理相关函数实现

//** 检查指定的进程ID是否仍在运行
bool is_process_running(pid_t pid) {
    if (pid <= 0) {
        return false;
    }
    
    // 使用kill(pid, 0)检查进程是否存在
    // 如果进程存在，kill返回0；如果不存在，返回-1并设置errno为ESRCH
    if (kill(pid, 0) == 0) {
        return true;
    } else {
        // 检查错误类型
        if (errno == ESRCH) {
            // 进程不存在
            return false;
        } else if (errno == EPERM) {
            // 没有权限发送信号，但进程存在
            return true;
        } else {
            // 其他错误，假设进程不存在
            return false;
        }
    }
}

//** 获取进程的命令行信息
std::string get_process_cmdline(pid_t pid) {
    if (pid <= 0) {
        return "";
    }
    
    std::string proc_cmdline_path = "/proc/" + std::to_string(pid) + "/cmdline";
    std::ifstream cmdline_file(proc_cmdline_path);
    
    if (!cmdline_file.is_open()) {
        return "";
    }
    
    std::string cmdline;
    std::getline(cmdline_file, cmdline);
    cmdline_file.close();
    
    // /proc/pid/cmdline中的参数由NULL字符分隔，替换为空格
    for (char& c : cmdline) {
        if (c == '\0') {
            c = ' ';
        }
    }
    
    // 移除末尾多余的空格
    while (!cmdline.empty() && cmdline.back() == ' ') {
        cmdline.pop_back();
    }
    
    return cmdline;
}

//** 尝试终止指定的进程
bool terminate_process_gracefully(pid_t pid, const std::string& process_name) {
    (void)process_name; // 消除未使用参数警告
    if (pid <= 0) {
        return false;
    }
    
    // 首先检查进程是否仍在运行
    if (!is_process_running(pid)) {
        return true;
    }
    
    // 获取进程命令行信息用于确认
    std::string cmdline = get_process_cmdline(pid);
    
    // 检查命令行是否包含我们期望的证明器程序名（vampire、eprover、cse）
    bool is_expected_process = false;
    if (cmdline.find("vampire") != std::string::npos || 
        cmdline.find("eprover") != std::string::npos || 
        cmdline.find("cse") != std::string::npos) {
        is_expected_process = true;
    }
    
    if (!is_expected_process) {
        return false;
    }
    
    // 发送SIGTERM信号
    if (kill(pid, SIGTERM) != 0) {
        if (errno == ESRCH) {
            return true;
        } else {
            return false;
        }
    }
    
    // 等待进程退出，最多等待3秒
    const int max_wait_iterations = 30; // 30 * 100ms = 3秒
    for (int i = 0; i < max_wait_iterations; ++i) {
        if (!is_process_running(pid)) {
            return true;
        }
        usleep(100000); // 等待100ms
    }
    
    // 如果进程仍未退出，发送SIGKILL信号强制终止
    if (kill(pid, SIGKILL) != 0) {
        if (errno == ESRCH) {
            return true;
        } else {
            return false;
        }
    }
    
    // 再等待1秒确认进程被强制终止
    for (int i = 0; i < 10; ++i) {
        if (!is_process_running(pid)) {
            return true;
        }
        usleep(100000); // 等待100ms
    }
    
    return false;
}

//** 获取指定进程的所有子进程PID
std::vector<pid_t> get_child_processes(pid_t parent_pid) {
    std::vector<pid_t> child_pids;
    
    if (parent_pid <= 0) {
        return child_pids;
    }
    
    // 读取/proc/[parent_pid]/task/[parent_pid]/children文件
    std::string children_file = "/proc/" + std::to_string(parent_pid) + "/task/" + std::to_string(parent_pid) + "/children";
    std::ifstream file(children_file);
    
    if (file.is_open()) {
        std::string line;
        if (std::getline(file, line)) {
            std::istringstream iss(line);
            pid_t child_pid;
            while (iss >> child_pid) {
                if (child_pid > 0) {
                    child_pids.push_back(child_pid);
                }
            }
        }
        file.close();
    } else {
        // 如果children文件不存在，尝试通过/proc目录扫描查找子进程
        DIR* proc_dir = opendir("/proc");
        if (proc_dir != nullptr) {
            struct dirent* entry;
            while ((entry = readdir(proc_dir)) != nullptr) {
                // 检查是否为数字目录名（进程ID）
                if (entry->d_type == DT_DIR && std::isdigit(entry->d_name[0])) {
                    pid_t pid = std::atoi(entry->d_name);
                    if (pid > 0) {
                        // 检查该进程的父进程ID
                        std::string stat_file = "/proc/" + std::to_string(pid) + "/stat";
                        std::ifstream stat_stream(stat_file);
                        if (stat_stream.is_open()) {
                            std::string line;
                            if (std::getline(stat_stream, line)) {
                                std::istringstream iss(line);
                                std::string field;
                                // stat文件的第4个字段是父进程ID
                                for (int i = 0; i < 4 && iss >> field; ++i) {
                                    if (i == 3) { // 第4个字段（从0开始计数）
                                        pid_t ppid = std::atoi(field.c_str());
                                        if (ppid == parent_pid) {
                                            child_pids.push_back(pid);
                                        }
                                    }
                                }
                            }
                            stat_stream.close();
                        }
                    }
                }
            }
            closedir(proc_dir);
        }
    }
    
    return child_pids;
}

//** 递归获取进程树中的所有进程PID
std::vector<pid_t> get_process_tree(pid_t root_pid) {
    std::vector<pid_t> all_pids;
    std::queue<pid_t> pid_queue;
    std::set<pid_t> visited; // 防止循环引用
    
    if (root_pid <= 0) {
        return all_pids;
    }
    
    pid_queue.push(root_pid);
    visited.insert(root_pid);
    
    while (!pid_queue.empty()) {
        pid_t current_pid = pid_queue.front();
        pid_queue.pop();
        
        // 检查进程是否仍然存在
        if (is_process_running(current_pid)) {
            all_pids.push_back(current_pid);
            
            // 获取当前进程的所有子进程
            std::vector<pid_t> children = get_child_processes(current_pid);
            for (pid_t child_pid : children) {
                if (visited.find(child_pid) == visited.end()) {
                    visited.insert(child_pid);
                    pid_queue.push(child_pid);
                }
            }
        }
    }
    
    return all_pids;
}

//** 终止整个进程树
int terminate_process_tree(pid_t root_pid, const std::string& process_name) {
    (void)process_name; // 消除未使用参数警告
    if (root_pid <= 0) {
        return 0;
    }
    
    // 开始清理进程树
    
    // 获取整个进程树
    std::vector<pid_t> process_tree = get_process_tree(root_pid);
    
    if (process_tree.empty()) {
        return 0;
    }
    
    // 分析进程树结构，但不输出详细信息
    
    int terminated_count = 0;
    
    // 逆序终止进程（先终止子进程，再终止父进程）
    std::reverse(process_tree.begin(), process_tree.end());
    
    // 第一轮：发送SIGTERM信号给所有进程
    for (pid_t pid : process_tree) {
        if (is_process_running(pid)) {
            kill(pid, SIGTERM);
        }
    }
    
    // 等待进程优雅退出
    const int max_wait_time = 3000; // 3秒
    const int check_interval = 100;  // 100ms
    
    for (int elapsed = 0; elapsed < max_wait_time; elapsed += check_interval) {
        bool all_terminated = true;
        for (pid_t pid : process_tree) {
            if (is_process_running(pid)) {
                all_terminated = false;
                break;
            }
        }
        
        if (all_terminated) {
            terminated_count = process_tree.size();
            return terminated_count;
        }
        
        usleep(check_interval * 1000); // 转换为微秒
    }
    
    // 第二轮：对仍在运行的进程发送SIGKILL信号
    std::vector<pid_t> remaining_processes;
    
    for (pid_t pid : process_tree) {
        if (is_process_running(pid)) {
            remaining_processes.push_back(pid);
            kill(pid, SIGKILL);
        }
    }
    
    // 最终检查
    usleep(500000); // 等待500ms
    for (pid_t pid : process_tree) {
        if (!is_process_running(pid)) {
            terminated_count++;
        }
    }
    
    // 进程树清理完成
    
    return terminated_count;
}

//** 检测并清理程序产生的所有孤儿进程（包括子进程树）
int detect_and_cleanup_orphan_processes(const std::vector<ProcessResult*>& process_results) {
    // 孤儿进程检测和清理，但不输出详细信息
    
    int orphan_count = 0;
    int total_processes = 0;
    int running_processes = 0;
    int terminated_processes = 0;
    
    // 遍历所有进程结果
    for (const auto& result : process_results) {
        if (!result) {
            continue;
        }
        
        total_processes++;
        
        // 检查进程ID是否有效
        if (result->vampire_process_id <= 0) {
            continue;
        }
        
        std::string process_name;
        if (result->process_id == 0) {
            process_name = "基准进程";
        } else {
            process_name = "进程" + std::to_string(result->process_id);
        }
        
        // 检查进程是否仍在运行
        if (is_process_running(result->vampire_process_id)) {
            orphan_count++;
            running_processes++;
            
            // 首先检查是否有子进程
            std::vector<pid_t> process_tree = get_process_tree(result->vampire_process_id);
            if (process_tree.size() > 1) {
                // 使用进程树清理功能
                int tree_terminated = terminate_process_tree(result->vampire_process_id, process_name);
                if (tree_terminated > 0) {
                    terminated_processes++;
                }
            } else {
                // 单个进程，使用原有的终止方法

                
                if (terminate_process_gracefully(result->vampire_process_id, process_name)) {
                    terminated_processes++;
                }
            }
        }
    }
    
    // 检测结果汇总完成，但不输出详细信息
    
    return orphan_count;
}

// --- 证明输出生成功能实现 ---

void generate_proof_output(const std::string& original_problem_path,
                          const std::string& cnf_file_path,
                          const std::vector<Lemma>& lemmas,
                          const std::string& prover_result_file,
                          bool is_baseline_proof,
                          const std::string& prover_name,
                          double total_time_seconds,
                          const std::string& problem_name) {
    
    // 获取问题文件名（去掉路径和.p后缀）
    std::string problem_filename = original_problem_path;
    size_t last_slash = problem_filename.find_last_of("/\\");
    if (last_slash != std::string::npos) {
        problem_filename = problem_filename.substr(last_slash + 1);
    }
    
    // 只移除结尾的.p扩展名，保留文件名中的其他点号
    if (problem_filename.size() >= 2 && problem_filename.substr(problem_filename.size() - 2) == ".p") {
        problem_filename = problem_filename.substr(0, problem_filename.size() - 2);
    }
    
    // 检测结果文件中的实际SZS状态
    std::string actual_szs_status = "Theorem"; // 默认为Theorem
    std::string output_section_name = "Proof"; // 默认为Proof
    
    if (file_exists(prover_result_file)) {
        std::ifstream result_file(prover_result_file);
        std::string line;
        while (std::getline(result_file, line)) {
            if (line.find("# SZS status Satisfiable") != std::string::npos) {
                actual_szs_status = "Satisfiable";
                output_section_name = "Saturation";
                break;
            } else if (line.find("# SZS status Theorem") != std::string::npos ||
                      line.find("# SZS status Unsatisfiable") != std::string::npos) {
                actual_szs_status = "Theorem";
                output_section_name = "Proof";
                break;
            }
        }
        result_file.close();
    }
    
    // 输出版本和问题信息
    std::cout << "# Version  : CSE_E---1.7" << std::endl;
    std::cout << "# Problem  : " << problem_filename << ".p" << std::endl;
    std::cout << "# Proof found" << std::endl;
    std::cout << "# SZS status " << actual_szs_status << " for " << problem_filename << ".p" << std::endl;
    
    // CSE特殊处理：输出CNF信息和.out文件内容
    if (prover_name == "CSE") {
        std::cout << "# SZS output start " << output_section_name << std::endl;
        
        // 1. 输出CNF转换（包含完整inference信息）
        output_cnf_conversion(cnf_file_path, original_problem_path);
        
        // 2. 输出推导过程标题和CSE的.out文件内容
        std::cout << "# ========== " << output_section_name << " Derivation ==========" << std::endl;
        
        // 使用传入的problem_name，如果为空则使用problem_filename
        std::string cse_problem_name = problem_name.empty() ? problem_filename : problem_name;
        output_cse_out_file_content(cse_problem_name);
        
        std::cout << "# SZS output end " << output_section_name << std::endl;
        std::cout << "# Total time : " << std::fixed << std::setprecision(3) << total_time_seconds << "s" << std::endl;
    }
    else if (prover_name == "E-Prover") {
        std::cout << "# SZS output start " << output_section_name << std::endl;
        
        // 1. 输出CNF转换
        output_cnf_conversion(cnf_file_path, original_problem_path);
        
        // 2. 如果不是基准进程证明，输出引理
        if (!is_baseline_proof && !lemmas.empty()) {
            std::cout << "# ========== Added Lemmas ==========" << std::endl;
            output_lemmas(lemmas);
        }
        
        // 3. 输出推导过程
        std::cout << "# ========== " << output_section_name << " Derivation ==========" << std::endl;
        output_derivation(prover_result_file);
        
        std::cout << "# SZS output end " << output_section_name << std::endl;
        std::cout << "# Total time : " << std::fixed << std::setprecision(3) << total_time_seconds << "s" << std::endl;
        } else {
        // 对于其他证明器，只输出基本信息
        std::cout << "# " << output_section_name << " completed by " << prover_name << std::endl;
        std::cout << "# Total time : " << std::fixed << std::setprecision(3) << total_time_seconds << "s" << std::endl;
    }
}

void output_cnf_conversion(const std::string& cnf_file_path, const std::string& original_problem_path) {
    // 检测原始问题文件是否已经是CNF格式
    auto is_already_cnf = [](const std::string& file_path) -> bool {
        // 方法1：根据TPTP文件名约定判断
        // 带-号的文件通常是CNF格式，带+号的是FOF格式
        std::string filename = file_path;
        size_t last_slash = filename.find_last_of("/\\");
        if (last_slash != std::string::npos) {
            filename = filename.substr(last_slash + 1);
        }
        
        // 检查文件名中是否包含-号（CNF格式标识）
        if (filename.find("-") != std::string::npos) {
            return true;
        }
        
        // 方法2：检查文件内容中是否已经包含cnf子句
        std::ifstream file(file_path);
        if (file.is_open()) {
            std::string line;
            int cnf_lines = 0;
            int fof_lines = 0;
            
            // 读取前20行进行判断
            for (int i = 0; i < 20 && std::getline(file, line); ++i) {
                if (line.find("cnf(") != std::string::npos) {
                    cnf_lines++;
                } else if (line.find("fof(") != std::string::npos) {
                    fof_lines++;
                }
            }
            file.close();
            
            // 如果有CNF子句且没有FOF子句，认为是CNF格式
            if (cnf_lines > 0 && fof_lines == 0) {
                return true;
            }
        }
        
        return false;
    };
    
    // 选择要输出的文件：优先使用full版本的CNF文件
    std::string source_file_path;
    
    // 首先尝试使用带有完整inference信息的_full.cnf文件
    std::string full_cnf_path = cnf_file_path;
    size_t dot_pos = full_cnf_path.find_last_of(".");
    if (dot_pos != std::string::npos) {
        full_cnf_path.insert(dot_pos, "_full");
    } else {
        full_cnf_path += "_full";
    }
    
    if (file_exists(full_cnf_path)) {
        source_file_path = full_cnf_path;
    }
    
    // 如果_full.cnf文件不存在或无效，根据原始问题类型选择文件
    if (source_file_path.empty()) {
        if (is_already_cnf(original_problem_path)) {
            // 原始问题已经是CNF格式，使用原始文件
            source_file_path = original_problem_path;
            std::cout << "# 使用原始CNF文件: " << source_file_path << std::endl;
        } else {
            // 回退到简化版本的CNF文件
            source_file_path = cnf_file_path;
            std::cout << "# 使用简化CNF文件: " << source_file_path << std::endl;
        }
    }
    
    if (!file_exists(source_file_path)) {
        std::cout << "# CNF file not found: " << source_file_path << std::endl;
        return;
    }
    
    std::ifstream cnf_file(source_file_path);
    std::string line;
    bool in_cnf_clause = false;
    bool in_fof_clause = false;
    std::string current_clause;
    
    while (std::getline(cnf_file, line)) {
        // 跳过所有%开头的注释行
        if (line.find("%") == 0) {
            continue;
        }
        
        // 检查是否开始一个新的CNF子句
        if (line.find("cnf(") == 0) {
            in_cnf_clause = true;
            current_clause = line;
            
            // 检查是否是单行CNF子句（包含).）
            if (line.find(").") != std::string::npos) {
                std::cout << current_clause << std::endl;
                in_cnf_clause = false;
                current_clause.clear();
            }
        }
        // 检查是否开始一个新的FOF子句
        else if (line.find("fof(") == 0) {
            in_fof_clause = true;
            current_clause = line;
            
            // 检查是否是单行FOF子句（包含).）
            if (line.find(").") != std::string::npos) {
                std::cout << current_clause << std::endl;
                in_fof_clause = false;
                current_clause.clear();
            }
        }
        // 如果正在处理多行CNF或FOF子句
        else if (in_cnf_clause || in_fof_clause) {
            current_clause += "\n" + line;
            
            // 检查是否是子句的结束（包含).）
            if (line.find(").") != std::string::npos) {
                std::cout << current_clause << std::endl;
                in_cnf_clause = false;
                in_fof_clause = false;
                current_clause.clear();
            }
        }
        // 处理其他非注释、非空行（如果需要的话）
        else if (!line.empty() && line.find("%") != 0) {
            // 对于不是CNF或FOF开头但也不是注释的行，可以选择输出或忽略
            // 这里选择忽略，只输出CNF和FOF子句
        }
    }
    cnf_file.close();
}

void output_lemmas(const std::vector<Lemma>& lemmas) {
    std::cout << "# Total lemmas added: " << lemmas.size() << std::endl;
    
    // 输出引理
    for (const auto& lemma : lemmas) {
        if (lemma.derivation_type == "cse" && !lemma.original_tptp_line.empty()) {
            // 对于CSE引理，如果有保存的原始TPTP行，直接使用
            std::cout << lemma.original_tptp_line << std::endl;
    } else {
            // 对于其他引理，使用转换后的TPTP格式
            if (!lemma.content.empty()) {
                std::cout << lemma.to_tptp_cnf() << std::endl;
            }
        }
    }
}

void output_derivation(const std::string& prover_result_file) {
    if (!file_exists(prover_result_file)) {
        std::cout << "# Prover result file not found: " << prover_result_file << std::endl;
        return;
    }
    
    std::vector<std::string> derivation = extract_derivation_from_eprover(prover_result_file);
    
    for (const auto& line : derivation) {
        std::cout << line << std::endl;
    }
}

void output_cse_out_file_content(const std::string& problem_name) {
    // 构建CSE输出目录路径
    std::string output_base = "./output/";
    std::string output_dir;
    
    // 查找以问题名开头的输出目录
    std::string find_cmd = "find \"" + output_base + "\" -maxdepth 1 -type d -name \"" + problem_name + "*\" 2>/dev/null | head -1";
    FILE* pipe = popen(find_cmd.c_str(), "r");
    if (pipe) {
        char buffer[1024];
        if (fgets(buffer, sizeof(buffer), pipe) != nullptr) {
            output_dir = std::string(buffer);
            // 移除换行符
            if (!output_dir.empty() && output_dir.back() == '\n') {
                output_dir.pop_back();
            }
        }
        pclose(pipe);
    }
    
    // 如果找不到现有目录，使用默认的eq_rstfp后缀作为预期目录
    if (output_dir.empty()) {
        output_dir = output_base + problem_name + "_short+eq_rstfp";
    }
    
    // 查找目录中的.out文件
    std::string cse_output_file;
    std::string find_out_cmd = "find \"" + output_dir + "\" -maxdepth 1 -name \"*.out\" 2>/dev/null | head -1";
    pipe = popen(find_out_cmd.c_str(), "r");
    if (pipe) {
        char buffer[1024];
        if (fgets(buffer, sizeof(buffer), pipe) != nullptr) {
            cse_output_file = std::string(buffer);
            // 移除换行符
            if (!cse_output_file.empty() && cse_output_file.back() == '\n') {
                cse_output_file.pop_back();
            }
        }
        pclose(pipe);
    }
    
    // 如果没找到，使用默认路径
    if (cse_output_file.empty()) {
        cse_output_file = output_dir + "/" + problem_name + "_short+eq_rstfp.out";
    }
    
    // 检查文件是否存在
    if (!file_exists(cse_output_file)) {
        std::cout << "# CSE .out file not found: " << cse_output_file << std::endl;
        return;
    }
    
    // 读取并输出.out文件的全部内容
    std::ifstream out_file(cse_output_file);
    if (!out_file.is_open()) {
        std::cout << "# Failed to open CSE .out file: " << cse_output_file << std::endl;
        return;
    }
    
    std::string line;
    while (std::getline(out_file, line)) {
        std::cout << line << std::endl;
    }
    
    out_file.close();
}

std::vector<std::string> extract_derivation_from_eprover(const std::string& result_file) {
    std::vector<std::string> derivation;
    std::ifstream file(result_file);
    std::string line;
    bool in_derivation = false;
    bool found_section_start = false;
    
    if (!file.is_open()) {
        return derivation;
    }
    
    while (std::getline(file, line)) {
        // 查找推导过程开始标志（支持CNFRefutation和Saturation两种）
        if (line.find("# SZS output start CNFRefutation") != std::string::npos ||
            line.find("# SZS output start Saturation") != std::string::npos) {
            found_section_start = true;
            in_derivation = true;
            continue;
        }
        
        // 查找推导过程结束标志（支持CNFRefutation和Saturation两种）
        if (found_section_start && 
            (line.find("# SZS output end CNFRefutation") != std::string::npos ||
             line.find("# SZS output end Saturation") != std::string::npos)) {
            break;
        }
        
        // 如果在推导过程中，收集相关行
        if (in_derivation) {
            // 跳过某些不需要的行
            if (line.find("# Proof found!") != std::string::npos ||
                line.find("# No proof found!") != std::string::npos ||
                line.find("# SZS status") != std::string::npos ||
                line.empty()) {
                continue;
            }
            
            // 收集fof和cnf子句
            if (line.find("fof(") == 0 || line.find("cnf(") == 0) {
                derivation.push_back(line);
            }
        }
    }
    
    file.close();
    return derivation;
}

// 从问题文件中提取引理集合
std::vector<Lemma> extract_lemmas_from_problem_file(const std::string& problem_file_path) {
    std::vector<Lemma> lemmas;
    
    if (!file_exists(problem_file_path)) {
        return lemmas;
    }
    
    std::ifstream file(problem_file_path);
    if (!file.is_open()) {
        return lemmas;
    }
    
    std::string line;
    bool in_lemma_section = false;
    
    while (std::getline(file, line)) {
        // 检查是否进入引理区域
        if (line.find("% 自动添加的引理") != std::string::npos) {
            in_lemma_section = true;
            continue;
        }
        
        // 如果在引理区域中，解析CNF引理
        if (in_lemma_section && !line.empty() && line.find("cnf(") == 0) {
            // 从CNF行重建引理对象
            Lemma lemma;
            
            // 解析CNF格式：cnf(id, role, (content)).
            size_t id_start = line.find("cnf(") + 4;
            size_t first_comma = line.find(",", id_start);
            size_t second_comma = line.find(",", first_comma + 1);
            size_t content_start = line.find("(", second_comma);
            size_t content_end = line.find(").", content_start);
            
            if (first_comma != std::string::npos && second_comma != std::string::npos && 
                content_start != std::string::npos && content_end != std::string::npos) {
                
                // 提取ID
                lemma.id = line.substr(id_start, first_comma - id_start);
                // 去除空格
                lemma.id.erase(0, lemma.id.find_first_not_of(" \t"));
                lemma.id.erase(lemma.id.find_last_not_of(" \t") + 1);
                
                // 提取role
                lemma.role = line.substr(first_comma + 1, second_comma - first_comma - 1);
                lemma.role.erase(0, lemma.role.find_first_not_of(" \t"));
                lemma.role.erase(lemma.role.find_last_not_of(" \t") + 1);
                
                // 提取content
                lemma.content = line.substr(content_start + 1, content_end - content_start - 1);
                lemma.content.erase(0, lemma.content.find_first_not_of(" \t"));
                lemma.content.erase(lemma.content.find_last_not_of(" \t") + 1);
                
                // 检查是否是CSE引理，如果是则保存原始TPTP行
                if (lemma.id.find("i_0_") == 0) {
                    // 以i_0_开头的是CSE引理
                    lemma.derivation_type = "cse";
                    lemma.original_tptp_line = line;  // 保存完整的原始行
                } else if (lemma.id.find("cseLemma") == 0) {
                    // 旧格式的CSE引理ID
                    lemma.derivation_type = "cse";
                    lemma.original_tptp_line = line;  // 保存完整的原始行
                } else if (lemma.id.find("vampireLemma") == 0) {
                    lemma.derivation_type = "vampire";
                }
                
                // 初始化其他字段
                lemma.clause_number = -1;
                lemma.derivation_depth = -1;
                
                lemmas.push_back(lemma);
            }
        }
    }
    
    file.close();
    return lemmas;
}