//** 引理划分器模块 - 头文件
//** 提供多种引理划分策略的实现
//** 支持轮询分配、年龄分层、推导深度分层和深度均衡等策略

#ifndef LEMMA_DIVIDERS_H
#define LEMMA_DIVIDERS_H

#include "lemma_structures.h"



//** 创建指定策略的引理划分器
//** @param strategy_name 策略名称
//** @return 对应的引理划分器实例
std::unique_ptr<LemmaDivider> createLemmaDivider(const std::string& strategy_name);

//** 轮询分配划分器
//** 使用轮询(round-robin)方式将引理平均分配到k个集合中
class RoundRobinDivider : public LemmaDivider {
public:
    std::vector<std::vector<Lemma>> divide(const std::vector<Lemma>& lemmas, int num_sets) override;
    std::string getName() const override;
    std::string getDescription() const override;
};

//** 年龄分层划分器
//** 根据子句序号（clause number）将引理分为不同年龄段
//** 较小的序号表示较早生成的引理，每个集合包含一个年龄段的引理
class AgeBasedDivider : public LemmaDivider {
public:
    std::vector<std::vector<Lemma>> divide(const std::vector<Lemma>& lemmas, int num_sets) override;
    std::string getName() const override;
    std::string getDescription() const override;
};

//** 推导深度分层划分器
//** 根据引理的推导深度（从input子句开始的推理步数）将引理分层
//** 每个集合包含相似推导深度的引理
class DepthBasedDivider : public LemmaDivider {
public:
    std::vector<std::vector<Lemma>> divide(const std::vector<Lemma>& lemmas, int num_sets) override;
    
    //** 支持单元子句过滤的divide方法重载
    //** @param lemmas 要划分的引理列表
    //** @param num_sets 要分割的集合数量
    //** @param unit_clauses_only 是否只保留单元子句
    //** @return 划分后的引理集合向量
    std::vector<std::vector<Lemma>> divide(const std::vector<Lemma>& lemmas, int num_sets, bool unit_clauses_only);
    
    std::string getName() const override;
    std::string getDescription() const override;
};

//** 深度均衡划分器
//** 基于推导深度，但确保每个集合都包含各种深度的子句
//** 在每个深度层内使用轮询方式分配，使深度分布更加均匀
class DepthBalancedDivider : public LemmaDivider {
public:
    std::vector<std::vector<Lemma>> divide(const std::vector<Lemma>& lemmas, int num_sets) override;
    
    //** 支持单元子句过滤的divide方法重载
    //** @param lemmas 要划分的引理列表
    //** @param num_sets 要分割的集合数量
    //** @param unit_clauses_only 是否只保留单元子句
    //** @return 划分后的引理集合向量
    std::vector<std::vector<Lemma>> divide(const std::vector<Lemma>& lemmas, int num_sets, bool unit_clauses_only);
    
    std::string getName() const override;
    std::string getDescription() const override;
};



//** 显示所有可用的划分策略
void showAvailableDivisionStrategies();

//** 轮询分配策略
//** @param selected_lemmas 已过滤的引理列表
//** @param k_sets 要分割的集合数量
//** @return k个引理集合的向量，每个集合包含部分引理
std::vector<std::vector<Lemma>> divide_lemmas_round_robin(const std::vector<Lemma>& selected_lemmas, int k_sets);

#endif // LEMMA_DIVIDERS_H 