/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
#include "globalFunc.h"

#include <iostream>
#include <string>
#include <vector>
#include <unordered_map>
#include <sstream>
#include <stdexcept>
#include <cctype>
#include <algorithm>
#include <utility>
#include <cstddef>
#include <iomanip>
#include <cstdlib>
#include <fstream>
#include <cstdio>
#include <regex>
#include <map>

CseSymbolMapper cseSymbolMapper;

// --- CNF 数据结构 ---

// 表示一个项 (变量、常量或函数应用)
struct Term {
    std::string name; // 名称
    std::vector<Term> args; // 参数
    bool is_variable = false; // 是否为变量

    // 默认构造函数
    Term() = default;

    // 变量/常量的构造函数
    Term(std::string n, bool is_var = false) : name(std::move(n)), is_variable(is_var) {
        if (!is_variable && !name.empty() && std::isupper(static_cast<unsigned char>(name[0]))) {
            is_variable = true;
        }
        if (!name.empty()) {
            bool is_number = true;
            bool first_char = true;
            for(char c : name) {
                if (first_char && c == '-') { first_char = false; continue; }
                if (!std::isdigit(static_cast<unsigned char>(c))) {
                    is_number = false;
                    break;
                }
                first_char = false;
            }
            if(is_number && !name.empty() && (name != "-" || name.length() > 1)) is_variable = false;
        }
        if (name == "=") is_variable = false;
    }

    // 函数应用的构造函数
    Term(std::string n, std::vector<Term> a) : name(std::move(n)), args(std::move(a)), is_variable(false) {
        if (name == "=") is_variable = false;
    }
    
    // 移动构造函数
    Term(Term&& other) noexcept :
        name(std::move(other.name)),
        args(std::move(other.args)),
        is_variable(other.is_variable)
    {}
    
     // 修正的移动赋值运算符
    Term& operator=(Term&& other) noexcept {
        name = std::move(other.name);
        args = std::move(other.args);
        is_variable = other.is_variable;
        return *this;
    }

//    // 支持移动构造和移动赋值，以避免拷贝问题
//    Term(Term&& other) noexcept = default;
//    Term& operator=(Term&& other) noexcept = default;

    // 禁用拷贝构造和拷贝赋值
    Term(const Term&) = delete;
    Term& operator=(const Term&) = delete;


    bool isConstant() const {
        return !is_variable && args.empty();
    }
};

// 表示一个文字 (正原子或负原子)
struct Literal {
    bool is_negated; // 是否被否定
    Term atom; // 表示 P(t1,...,tn) 或 =(t1, t2)，后者存储为 Term("=", {t1, t2})

    // 使用移动构造函数
    Literal(bool neg, Term&& a) : is_negated(neg), atom(std::move(a)) {}

    // 支持移动
    Literal(Literal&& other) noexcept = default;
    Literal& operator=(Literal&& other) noexcept = default;

    // 禁用拷贝
    Literal(const Literal&) = delete;
    Literal& operator=(const Literal&) = delete;
};

// 表示一个子句 (文字的析取)
// 使用移动语义来构建子句
using Clause = std::vector<Literal>;


// --- 解析辅助函数---


// 检查给定位置的操作符是否在括号和引号的顶层。
// 用于区分如 P(a=b) 中的 '=' 和 顶层的 a=b。
bool is_operator_toplevel(const std::string& s, size_t op_pos) {
    int balance = 0; // 括号平衡计数
    bool in_single_quotes = false;
    bool in_double_quotes = false;

    for (size_t i = 0; i < op_pos; ++i) {
        char current_char = s[i];

        // 处理引号状态切换，考虑转义字符
        if (current_char == '\'' && !in_double_quotes) {
            // 检查是否是转义的单引号
            if (i > 0 && s[i - 1] == '\\') {
                size_t backslash_count = 0;
                size_t j = i - 1;
                while (j > 0 && s[j] == '\\') {
                    backslash_count++;
                    j--;
                }
                // 如果反斜杠数量是奇数，则引号被转义，不改变状态
                if (backslash_count % 2 == 1) {
                    continue;
                }
            }
            in_single_quotes = !in_single_quotes;
        } else if (current_char == '"' && !in_single_quotes) {
            // 检查是否是转义的双引号
            if (i > 0 && s[i - 1] == '\\') {
                size_t backslash_count = 0;
                size_t j = i - 1;
                while (j > 0 && s[j] == '\\') {
                    backslash_count++;
                    j--;
                }
                // 如果反斜杠数量是奇数，则引号被转义，不改变状态
                if (backslash_count % 2 == 1) {
                    continue;
                }
            }
            in_double_quotes = !in_double_quotes;
        } else if (!in_single_quotes && !in_double_quotes) {
            // 不在引号内时，处理括号平衡
            if (current_char == '(') {
                balance++;
            } else if (current_char == ')') {
                if (balance > 0) {
                    balance--;
                } else {
                    // 遇到不匹配的右括号，说明操作符不在顶层
                    return false;
                }
            }
        }
    }
    // 最终括号平衡且不在引号内，则操作符在顶层
    bool result = (balance == 0 && !in_single_quotes && !in_double_quotes);
    return result;
}

// 跳过空白字符
void skip_whitespace(const std::string& s, size_t& pos) {
    while (pos < s.length() && std::isspace(static_cast<unsigned char>(s[pos]))) {
        pos++;
    }
}

// 查找给定起始括号索引对应的匹配结束括号，返回匹配括号的索引，如果未找到或格式错误则返回 -1
// 正确处理引号内的括号，不将其计入平衡计算
// 先进行快速扫描，检查是否有引号和转义字符
// 如果有引号和转义字符，则进行完整处理
int find_matching_paren(const std::string& s, size_t start_pos) {
    if (start_pos >= s.length() || s[start_pos] != '(') {
        throw std::runtime_error("find_matching_paren: start_pos 不是 '('。");
    }
    bool has_quotes = false;
    bool has_escapes = false;
    // 快速扫描检查是否有引号和转义字符
    for (size_t i = start_pos + 1; i < s.length(); ++i) {
        char c = s[i];
        if (c == '\'' || c == '"') {
            has_quotes = true;
            break;
        }
        if (c == '\\') {
            has_escapes = true;
            break;
        }
    }

    // === 无引号无转义的简单括号匹配 ===
    if (!has_quotes && !has_escapes) {
        int balance = 1;
        for (size_t i = start_pos + 1; i < s.length(); ++i) {
            if (s[i] == '(') {
                balance++;
            } else if (s[i] == ')') {
                balance--;
                if (balance == 0) {
                    return static_cast<int>(i);
                }
            }
        }
        return -1; // 未找到匹配的括号
    }
    // 后向搜索 ).
    if (has_quotes && !has_escapes) {
        // 从后向前查找可能的结束模式
        for (size_t i = s.length(); i > start_pos + 1; --i) {
            size_t pos = i - 1;
            if (s[pos] == ')') {
                // 检查这个 ')' 前面是否紧跟 ").'" 模式
                bool is_cnf_pattern = false;
                if (pos >= 2 && s[pos-1] == '"' && s[pos-2] == '.') {
                    is_cnf_pattern = true;
                } else if (pos >= 2 && s[pos-1] == '\'' && s[pos-2] == '.') {
                    is_cnf_pattern = true;
                }

                if (is_cnf_pattern) {
                    // 验证这个 ')' 是否真的匹配起始的 '('
                    int balance = 1;
                    bool in_single_quotes = false;
                    bool in_double_quotes = false;

                    for (size_t j = start_pos + 1; j < pos; ++j) {
                        char current_char = s[j];

                        if (current_char == '\'' && !in_double_quotes) {
                            in_single_quotes = !in_single_quotes;
                        } else if (current_char == '"' && !in_single_quotes) {
                            in_double_quotes = !in_double_quotes;
                        } else if (!in_single_quotes && !in_double_quotes) {
                            if (current_char == '(') {
                                balance++;
                            } else if (current_char == ')') {
                                balance--;
                                if (balance == 0) {
                                    // 找到了更早的匹配
                                    break;
                                }
                            }
                        }
                    }

                    if (balance == 1) {
                        // 这个 ')' 匹配起始的 '('
                        return static_cast<int>(pos);
                    }
                }
            }
        }
    }

    //处理复杂情况（引号 + 转义字符）
    int balance = 1; // 起始括号计数为 1
    bool in_single_quotes = false;
    bool in_double_quotes = false;

    for (size_t i = start_pos + 1; i < s.length(); ++i) {
        char current_char = s[i];

        // 处理引号状态切换，考虑转义字符
        if (current_char == '\'' && !in_double_quotes) {
            // 检查是否是转义的单引号
            if (i > 0 && s[i - 1] == '\\') {
                size_t backslash_count = 0;
                size_t j = i - 1;
                while (j > 0 && s[j] == '\\') {
                    backslash_count++;
                    j--;
                }
                // 如果反斜杠数量是奇数，则引号被转义，不改变状态
                if (backslash_count % 2 == 1) {
                    continue;
                }
            }
            in_single_quotes = !in_single_quotes;
        } else if (current_char == '"' && !in_single_quotes) {
            // 检查是否是转义的双引号
            if (i > 0 && s[i - 1] == '\\') {
                size_t backslash_count = 0;
                size_t j = i - 1;
                while (j > 0 && s[j] == '\\') {
                    backslash_count++;
                    j--;
                }
                // 如果反斜杠数量是奇数，则引号被转义，不改变状态
                if (backslash_count % 2 == 1) {
                    continue;
                }
            }
            in_double_quotes = !in_double_quotes;
        } else if (!in_single_quotes && !in_double_quotes) {
            // 不在引号内时，处理括号平衡
            if (current_char == '(') {
                balance++;
            } else if (current_char == ')') {
                balance--;
                if (balance == 0) {
                    return static_cast<int>(i); // 找到匹配的括号
                }
            }
        }
    }
    return -1; // 未找到匹配的括号
}

// 查找匹配的引号，处理转义字符和复杂内容
// 返回匹配引号的索引，如果未找到则返回 string::npos
size_t find_matching_quote(const std::string& s, size_t start_pos, char quote_char) {
    if (start_pos >= s.length() || s[start_pos] != quote_char) {
        return std::string::npos;
    }

    for (size_t i = start_pos + 1; i < s.length(); ++i) {
        if (s[i] == quote_char) {
            // 检查是否是转义的引号
            if (i > start_pos + 1 && s[i - 1] == '\\') {
                // 需要检查反斜杠本身是否被转义
                size_t backslash_count = 0;
                size_t j = i - 1;
                while (j > start_pos && s[j] == '\\') {
                    backslash_count++;
                    j--;
                }
                // 如果反斜杠数量是奇数，则引号被转义，继续查找
                if (backslash_count % 2 == 1) {
                    continue;
                }
            }
            // 找到匹配的引号
            return i;
        }
    }

    return std::string::npos; // 未找到匹配的引号
}

// 前向声明
size_t find_matching_quote(const std::string& s, size_t start_pos, char quote_char);
int find_matching_paren(const std::string& s, size_t start_pos);
std::vector<Term> parse_arguments(const std::string& s, size_t& pos);
Term parse_term(const std::string& s, size_t& pos);
std::string restore_clause_body_optimized(const std::string& clause_body_str,
                                          const CseSymbolMapper& mapper);

// 从给定位置开始解析一个项 (变量、常量、函数应用)
Term parse_term(const std::string& s, size_t& pos) {
    skip_whitespace(s, pos);
    size_t start = pos;
    std::string name;
    bool is_quoted = false;

    if (pos >= s.length()) throw std::runtime_error("解析项时意外到达字符串末尾");

    // 处理带引号的常量名
    if (s[pos] == '\'') {
        is_quoted = true;
        size_t end_quote_pos = find_matching_quote(s, pos, '\'');
        if (end_quote_pos == std::string::npos) {
            throw std::runtime_error("未找到匹配的结束单引号，起始位置: " + std::to_string(start) +
                                     "，字符串长度: " + std::to_string(s.length()) +
                                     "，当前位置: " + std::to_string(pos) +
                                     "，上下文: '" + s.substr(start, std::min(static_cast<size_t>(100), s.length() - start)) + "'");
        }
        name = s.substr(start, end_quote_pos - start + 1);
        pos = end_quote_pos + 1;
    } else if (s[pos] == '"') {
        is_quoted = true;
        size_t end_quote_pos = find_matching_quote(s, pos, '"');
        if (end_quote_pos == std::string::npos) throw std::runtime_error("未找到匹配的结束双引号，起始位置: " + std::to_string(start));
        name = s.substr(start, end_quote_pos - start + 1);
        pos = end_quote_pos + 1;
        // 处理字母数字或下划线开头的标识符 (变量、函数、常量) 或负数
    } else if (std::isalnum(static_cast<unsigned char>(s[pos])) || s[pos] == '_' || s[pos] == '-') {
        size_t name_start = start;
        // 特殊处理负号开头的数字
        if (s[start] == '-') {
            if (start + 1 < s.length() && std::isdigit(static_cast<unsigned char>(s[start+1]))) {
                pos++; // 跳过负号
                while (pos < s.length() && std::isdigit(static_cast<unsigned char>(s[pos]))) pos++; // 读取数字部分
            } else {
                // 负号开头的非数字标识符 (如函数名 -func)
                pos++;
                while (pos < s.length() && (std::isalnum(static_cast<unsigned char>(s[pos])) || s[pos] == '_')) pos++;
            }
        } else {
            // 普通标识符
            while (pos < s.length() && (std::isalnum(static_cast<unsigned char>(s[pos])) || s[pos] == '_')) pos++;
        }
        name = s.substr(name_start, pos - name_start);
        if (name.empty()) throw std::runtime_error("解析项名称失败，位置 " + std::to_string(start));
        // 处理等号 '=' (作为特殊函数/谓词名)
    } else if (pos < s.length() && s[pos] == '='){
        name = "=";
        pos++;
        // 处理以括号开头的表达式 (括号内的复合表达式)
    } else if (pos < s.length() && s[pos] == '(') {
        // 找到匹配的右括号
        int matching_paren = find_matching_paren(s, pos);
        if (matching_paren == -1) {
            throw std::runtime_error("未找到匹配的右括号，起始位置: " + std::to_string(pos));
        }

        // 提取括号内的内容
        std::string inner_content = s.substr(pos + 1, matching_paren - pos - 1);
        pos = matching_paren + 1;

        // 递归解析括号内的内容
        size_t inner_pos = 0;
        Term inner_term = parse_term(inner_content, inner_pos);
        return inner_term;
    } else {
        // 无效字符开头
        throw std::runtime_error("无效的项起始字符 '" + (pos < s.length() ? std::string(1, s[pos]) : "<EOS>") + "' 在位置 " + std::to_string(start));
    }

    size_t pos_before_args_check = pos;
    skip_whitespace(s, pos);
    // 检查后面是否有 '('，表示是函数应用
    if (pos < s.length() && s[pos] == '(') {
        pos++; // 跳过 '('
        std::vector<Term> args = parse_arguments(s, pos); // 递归解析参数列表
        skip_whitespace(s, pos);
        if (pos >= s.length() || s[pos] != ')') throw std::runtime_error("函数/谓词 '" + name + "' 的参数后缺少 ')'");
        pos++; // 跳过 ')'
        return Term(name, std::move(args)); // 返回函数应用 Term
    } else {
        // 没有 '(', 是常量或变量
        pos = pos_before_args_check; // 回退位置
        skip_whitespace(s, pos);
        if (is_quoted || name == "=") {
            // 带引号的或等号 '=' 本身是常量
            return Term(name, false);
        } else {
            // 其他情况根据首字母大写判断是否为变量
            return Term(name);
        }
    }
}

// 解析括号中的参数列表
std::vector<Term> parse_arguments(const std::string& s, size_t& pos) {
    std::vector<Term> args;
    skip_whitespace(s, pos);
    if (pos >= s.length()) throw std::runtime_error("解析参数时意外到达字符串末尾");

    // 如果紧接着是 ')'，表示空参数列表
    if (s[pos] == ')') return args;

    // 循环解析参数，直到遇到 ')'
    while (true) {
        Term arg = parse_term(s, pos); // 解析一个参数项
        args.push_back(std::move(arg));
        skip_whitespace(s, pos);

        if (pos >= s.length()) throw std::runtime_error("解析参数列表时，在参数后意外到达字符串末尾");

        if (s[pos] == ')') break; // 参数列表结束
        else if (s[pos] == ',') {
            // 遇到逗号，继续解析下一个参数
            pos++;
            skip_whitespace(s, pos);
            if (pos >= s.length()) throw std::runtime_error("解析参数列表时，在逗号后意外到达字符串末尾");
        } else {
            // 参数后既不是 ')' 也不是 ','，格式错误
            throw std::runtime_error("解析参数列表时，参数后缺少 ')' 或 ','，遇到 '" + std::string(1, s[pos]) + "'");
        }
    }
    return args;
}


// 解析一个文字 (可能是否定的原子或等式)
Literal parse_literal(std::string lit_str) {
    bool negated = false;
    // 检查并移除前导否定符号 '~'
    if (!lit_str.empty() && lit_str[0] == '~') {
        negated = true;
        lit_str = lit_str.substr(1);
    }

    // 清理文字字符串两端的空白
    size_t start_pos = lit_str.find_first_not_of(" \t\n\r");
    if (start_pos == std::string::npos) throw std::runtime_error("Literal string is empty or whitespace: " + lit_str);
    size_t end_pos = lit_str.find_last_not_of(" \t\n\r");
    lit_str = lit_str.substr(start_pos, end_pos - start_pos + 1);

    Term atom_term;
    bool parsed = false;
    size_t op_pos = std::string::npos;
    int op_len = 0;
    bool is_equality = false; // 标记是否为等式 (=)
    size_t search_pos = 0;

    // 优先查找顶层的 '!=' 操作符
    size_t neq_pos_check = std::string::npos;
    while ((neq_pos_check = lit_str.find("!=", search_pos)) != std::string::npos) {
        if (is_operator_toplevel(lit_str, neq_pos_check)) { // 确保 '!=' 不在括号或引号内
            op_pos = neq_pos_check;
            op_len = 2;
            is_equality = false; // '!=' 视为非等式
            parsed = true;
            break;
        } else {
            search_pos = neq_pos_check + 1; // 继续向后查找
        }
    }

    // 如果未找到 '!='，再查找顶层的 '=' 操作符
    if (!parsed) {
        search_pos = 0;
        size_t eq_pos_check = std::string::npos;
        while ((eq_pos_check = lit_str.find("=", search_pos)) != std::string::npos) {
            if (is_operator_toplevel(lit_str, eq_pos_check)) {
                // 排除前面是 '!' 的情况 (已经被 '!=' 处理)
                if (!(eq_pos_check > 0 && lit_str[eq_pos_check - 1] == '!')) {
                    op_pos = eq_pos_check;
                    op_len = 1;
                    is_equality = true; // '=' 视为等式
                    parsed = true;
                    break;
                }
            } else {
                search_pos = eq_pos_check + 1; // 继续向后查找
            }
        }
    }

    // 如果找到了顶层 '=' 或 '!='
    if (parsed) {
        Term lhs_term;
        Term rhs_term;
        size_t current_pos = 0;
        try {
            // 解析等式/不等式的左右两边
            lhs_term = parse_term(lit_str, current_pos);
            skip_whitespace(lit_str, current_pos);
            if (current_pos != op_pos) throw std::runtime_error("未能精确解析到操作符 '" + lit_str.substr(op_pos, op_len) + "' 之前的部分");

            current_pos += op_len; // 跳过操作符
            rhs_term = parse_term(lit_str, current_pos);
            skip_whitespace(lit_str, current_pos);
            if (current_pos != lit_str.length()) throw std::runtime_error("解析操作符 '" + lit_str.substr(op_pos, op_len) + "' 之后未能到达字符串末尾");

            // 将等式/不等式统一表示为 Term("=", {lhs, rhs})
            std::vector<Term> args;
            args.push_back(std::move(lhs_term));
            args.push_back(std::move(rhs_term));
            atom_term = Term("=", std::move(args));

            // 如果原始操作符是 '!='，则翻转否定标记
            if (!is_equality) negated = !negated;

        } catch (const std::exception& e) {
            // 如果解析等式/不等式失败，回退到按普通原子处理
            parsed = false;
            atom_term = Term(); // 重置 atom_term
        }
    }

    // 如果不是等式/不等式，或解析失败，则尝试按普通原子 (谓词应用) 解析
    if (!parsed) {
        size_t p = 0;
        atom_term = parse_term(lit_str, p);
        skip_whitespace(lit_str, p);
        // 确保整个字符串都被成功解析为一个 Term
        if(p == lit_str.length()) {
            parsed = true;
        } else {
            // 部分解析成功，说明格式可能错误
            parsed = false;
        }
    }

    // 如果两种方式都解析失败，则抛出错误
    if (!parsed) throw std::runtime_error("未能将文字字符串解析为任何已知结构: '" + lit_str + "'");
    // 返回最终构建的 Literal
    return Literal(negated, std::move(atom_term));
}


// 将 CNF 公式字符串解析为 Clause (文字的析取)
Clause parse_cnf_formula(const std::string& formula) {
    Clause clause;
    std::stringstream ss(formula);
    std::string segment;

    // 使用 '|' 作为分隔符，分割字符串得到各个文字片段
    while (std::getline(ss, segment, '|')) {
        if (segment.find_first_not_of(" \t\n\r") == std::string::npos) {
            continue; // 跳过空的或只包含空白的片段
        }

        clause.push_back(parse_literal(segment)); // 解析每个文字片段并添加到子句中
    }

    return clause; // 返回包含所有解析后文字的子句
}


// --- CSE 转换逻辑 ---


// CseSymbolMapper 类成员函数的实现
CseSymbolMapper::CseSymbolMapper() {
    // 预先定义等号映射
    predicate_map["="] = "E";
}

// 内部辅助函数实现
std::string CseSymbolMapper::get_mapping(const std::string& cnf_symbol,
                                         std::unordered_map<std::string, std::string>& map,
                                         const std::string& prefix,
                                         int& counter) {
    auto it = map.find(cnf_symbol);
    if (it == map.end()) {
        // 如果未找到映射，创建一个新的 CSE 符号并存储
        std::string cse_symbol = prefix + std::to_string(counter++);
        map[cnf_symbol] = cse_symbol;
        return cse_symbol;
    }
    // 如果已存在映射，直接返回
    return it->second;
}

std::string CseSymbolMapper::get_predicate_mapping(const std::string& cnf_symbol) {
    if (cnf_symbol == "=") {
        return "E"; // 特殊处理等号
    }
    // 调用内部辅助函数获取或创建谓词映射
    return get_mapping(cnf_symbol, predicate_map, "P", predicate_counter);
}

std::string CseSymbolMapper::get_function_mapping(const std::string& cnf_symbol) {
    // 调用内部辅助函数获取或创建函数映射
    return get_mapping(cnf_symbol, function_map, "f", function_counter);
}

std::string CseSymbolMapper::get_constant_mapping(const std::string& cnf_symbol) {
    // 调用内部辅助函数获取或创建常量映射
    return get_mapping(cnf_symbol, constant_map, "a", constant_counter);
}

// 实现访问器函数
const std::unordered_map<std::string, std::string>& CseSymbolMapper::getPredicateMap() const { return predicate_map; }
const std::unordered_map<std::string, std::string>& CseSymbolMapper::getFunctionMap() const { return function_map; }
const std::unordered_map<std::string, std::string>& CseSymbolMapper::getConstantMap() const { return constant_map; }

// 实现反向映射函数
std::unordered_map<std::string, std::string> CseSymbolMapper::getReversePredicateMap() const {
    std::unordered_map<std::string, std::string> reverse_map;
    for(const auto& pair : predicate_map) {
        if (pair.first == "=") continue; // 通常不需要反向映射 E
        reverse_map[pair.second] = pair.first;
    }
    return reverse_map;
}
std::unordered_map<std::string, std::string> CseSymbolMapper::getReverseFunctionMap() const {
    std::unordered_map<std::string, std::string> reverse_map;
    for(const auto& pair : function_map) {
        reverse_map[pair.second] = pair.first;
    }
    return reverse_map;
}
std::unordered_map<std::string, std::string> CseSymbolMapper::getReverseConstantMap() const {
    std::unordered_map<std::string, std::string> reverse_map;
    for(const auto& pair : constant_map) {
        reverse_map[pair.second] = pair.first;
    }
    return reverse_map;
}


// 将 CNF Term 转换为其 CSE 字符串表示 (内部函数)
std::string convert_term_to_cse(const Term& term,
                                CseSymbolMapper& mapper,
                                std::unordered_map<std::string, std::string>& local_variable_map,
                                int clause_id_number,
                                int& local_var_idx) {
    if (term.is_variable) {
        // 处理变量：查找局部映射，若无则创建新的 CSE 变量名
        auto it = local_variable_map.find(term.name);
        if (it == local_variable_map.end()) {
            std::string cse_var_name = "x" + std::to_string(clause_id_number) + std::to_string(local_var_idx++);
            local_variable_map[term.name] = cse_var_name;
            return cse_var_name;
        } else {
            return it->second; // 返回已存在的 CSE 变量名
        }
    } else if (term.args.empty()) {
        // 处理常量：使用 mapper 获取 CSE 常量名 (如 a1, a2)
        return mapper.get_constant_mapping(term.name);
    } else {
        // 处理函数应用或等式
        std::string cse_func_or_pred_name;
        if (term.name == "=") {
            cse_func_or_pred_name = mapper.get_predicate_mapping(term.name); // 得到 "E"
        } else {
            // 普通函数
            cse_func_or_pred_name = mapper.get_function_mapping(term.name); // 得到 "f1", "f2" 等
        }

        // 递归转换参数列表
        std::string cse_args_str;
        for (size_t i = 0; i < term.args.size(); ++i) {
            cse_args_str += convert_term_to_cse(term.args[i], mapper, local_variable_map, clause_id_number, local_var_idx);
            if (i < term.args.size() - 1) cse_args_str += ","; // 参数间用逗号分隔
        }
        // 组合函数/谓词名和参数列表
        return cse_func_or_pred_name + "(" + cse_args_str + ")"; // 如 E(a1,x1_1) 或 f1(a2,f2(a3))
    }
}

// 将 CNF Literal 转换为其 CSE 字符串表示 (内部函数)
std::string convert_literal_to_cse(const Literal& literal,
                                   CseSymbolMapper& mapper,
                                   std::unordered_map<std::string, std::string>& local_variable_map,
                                   int clause_id_number,
                                   int& local_var_idx) {
    std::string cse_atom_str;
    // 处理原子部分 (可能是普通谓词或等式)
    if (literal.atom.name == "=") {
        // convert_term_to_cse 会处理 "=" -> "E" 和参数的转换
        cse_atom_str = convert_term_to_cse(literal.atom, mapper, local_variable_map, clause_id_number, local_var_idx);
    } else {
        // 原子是普通谓词 P(...)
        std::string cse_pred_name = mapper.get_predicate_mapping(literal.atom.name); // 获取 CSE 谓词名 P1, P2 等
        std::string cse_args_str;

        // 处理无参谓词
        if (literal.atom.args.empty()) {
            //cse_args_str = "a99999"; // 直接使用特殊常量名
            cse_atom_str = cse_pred_name;
        } else {
            // 递归转换参数列表
            for (size_t i = 0; i < literal.atom.args.size(); ++i) {
                cse_args_str += convert_term_to_cse(literal.atom.args[i], mapper, local_variable_map, clause_id_number, local_var_idx);
                if (i < literal.atom.args.size() - 1) cse_args_str += ",";
            }
            cse_atom_str = cse_pred_name + "(" + cse_args_str + ")"; // 如 P1(a1, x1_1)
        }
    }
    // 根据 literal.is_negated 添加否定符号 '~'
    return (literal.is_negated ? "~" : "") + cse_atom_str; // 如 ~P1(a1, x1_1) 或 E(a1, x1_1)
}


// --- 核心转换函数实现 (在 .h 中声明) ---


// 修改函数名和参数，直接接收 CNF 文件路径
bool convert_cnf_to_cse(const std::string& input_cnf_path,
                        const std::string& output_cse_path,
                        CseSymbolMapper& mapper,
                        std::string& cnf_str,
                        std::vector<unsigned int>& goalIds) {
    // 1. 打开输入的 CNF 文件
    std::ifstream cnf_file(input_cnf_path); // 直接使用输入路径
    if (!cnf_file.is_open()) {
        std::cerr << "错误: 无法打开输入的 CNF 文件: " << input_cnf_path << std::endl;
        return false;
    }

    // 2. 打开输出 CSE 文件
    std::ofstream cse_output_file(output_cse_path);
    if (!cse_output_file.is_open()) {
        std::cerr << "错误: 无法打开用于写入的输出 CSE 文件: " << output_cse_path << std::endl;
        cnf_file.close(); // 关闭已打开的 cnf 文件
        return false;
    }

    std::string line;       // 用于存储读取的每一行
    unsigned int line_num = 0;       // 行号计数器
    bool success = true;    // 跟踪转换过程中是否发生错误

    // 3. 逐行读取 CNF，解析并转换为 CSE，写入输出文件
    while (std::getline(cnf_file, line)) {
        line_num++;
        try {
            // 跳过空行和注释行 (以 # 或 % 开头)
            size_t first_char_pos = line.find_first_not_of(" \t\n\r");
            if (first_char_pos == std::string::npos || line[first_char_pos] == '#' || line[first_char_pos] == '%') {
                continue;
            }
            // 只处理 cnf(...) 格式的行
            if (line.substr(first_char_pos, 4) != "cnf(") {
                std::cerr << "信息: 跳过非 cnf 行 " << line_num << ": " << line << std::endl;
                continue;
            }

            // --- 解析 cnf 行 ---
            // 查找必要的括号和逗号来提取 cnf ID 和公式部分
            size_t cnf_open_paren = line.find('(');
            if (cnf_open_paren == std::string::npos) { std::cerr << "警告: 行 " << line_num << ": 找不到 'cnf(' 的起始括号，跳过。" << std::endl; continue; }

            size_t first_comma_for_id = line.find(',', cnf_open_paren);
            if (first_comma_for_id == std::string::npos) { std::cerr << "警告: 行 " << line_num << ": 找不到 cnf ID 后的第一个逗号，跳过。" << std::endl; continue; }
            
            cnf_str += line + "\n";
            
            // 提取 cnf ID 字符串
            std::string cnf_id_str = line.substr(cnf_open_paren + 1, first_comma_for_id - (cnf_open_paren + 1));
            // 清理 cnf_id_str 的前后空格
            cnf_id_str.erase(0, cnf_id_str.find_first_not_of(" \t\n\r"));
            cnf_id_str.erase(cnf_id_str.find_last_not_of(" \t\n\r") + 1);

            // 从 cnf ID 中提取数字部分 (通常是最后一个下划线后面的数字)
            unsigned int clause_id_number = 0;
            size_t last_underscore = cnf_id_str.find_last_of('_');
            if (last_underscore == std::string::npos || last_underscore == cnf_id_str.length() - 1) {
                try {
                    clause_id_number = std::stoul(cnf_id_str);
                    std::cerr << "警告: 行 " << line_num << ": cnf ID '" << cnf_id_str << "' 格式非预期，尝试直接使用数字。" << std::endl;
                } catch (...) {
                    std::cerr << "警告: 行 " << line_num << ": cnf ID '" << cnf_id_str << "' 格式非预期且无法直接转为数字，将使用行号作为 ID: " << line_num << std::endl;
                    clause_id_number = line_num; // 使用行号作为备用 ID
                }
            } else {
                try {
                    // 提取最后一个下划线后的部分并转换为整数
                    clause_id_number = std::stoul(cnf_id_str.substr(last_underscore + 1));
                } catch (const std::invalid_argument& ia) {
                    std::cerr << "警告: 行 " << line_num << ": 无法从 cnf ID '" << cnf_id_str << "' 中解析数字 (" << ia.what() << ")，将使用行号作为 ID: " << line_num << std::endl;
                    clause_id_number = line_num; // 使用行号作为备用 ID
                } catch (const std::out_of_range& oor) {
                    std::cerr << "警告: 行 " << line_num << ": 从 cnf ID '" << cnf_id_str << "' 中解析的数字超出范围 (" << oor.what() << ")，将使用行号作为 ID: " << line_num << std::endl;
                    clause_id_number = line_num; // 使用行号作为备用 ID
                }
            }


            // 查找第二个逗号，以定位公式部分的开始
            size_t comma2_pos = line.find(',', first_comma_for_id + 1);
            if (comma2_pos == std::string::npos) { std::cerr << "警告: 行 " << line_num << ": 跳过缺少第二个逗号的cnf行。" << std::endl; continue; }

            //记录目标子句ID
            std::string tag_cla = line.substr(first_comma_for_id, comma2_pos - first_comma_for_id);
            if (tag_cla.find("negated_conjecture") != std::string::npos) {
                goalIds.push_back(clause_id_number);
            }

            // 使用 find_matching_paren 找到与 cnf 开始括号匹配的结束括号
            int matching_paren_pos = find_matching_paren(line, cnf_open_paren);
            if (matching_paren_pos == -1) {
                std::cerr << "警告: 行 " << line_num << ": 找不到与cnf开始括号匹配的结束括号，跳过。" << std::endl;
                continue;
            }
            size_t last_paren_pos = static_cast<size_t>(matching_paren_pos);

            if (last_paren_pos <= comma2_pos) { std::cerr << "警告: 行 " << line_num << ": 跳过格式错误(括号)的cnf行。" << std::endl; continue; }

            // 提取公式字符串 (位于第二个逗号和最后一个括号之间)
            std::string formula_str = line.substr(comma2_pos + 1, last_paren_pos - (comma2_pos + 1));

            // 清理 formula_str 的前后空格
            size_t first_non_space = formula_str.find_first_not_of(" \t\n\r");
            if (first_non_space == std::string::npos) formula_str = ""; // 如果全是空格
            else {
                size_t last_non_space = formula_str.find_last_not_of(" \t\n\r");
                formula_str = formula_str.substr(first_non_space, last_non_space - first_non_space + 1);
            }
            // 移除可能存在的外层括号 (eprover 有时会输出带括号的单个文字)
            if (!formula_str.empty() && formula_str.front() == '(' && formula_str.back() == ')') {
                // 需要检查括号是否匹配且确实是包裹整个公式的
                int matching_paren = find_matching_paren(formula_str, 0);
                if (matching_paren == static_cast<int>(formula_str.length() - 1)) {
                    formula_str = formula_str.substr(1, formula_str.length() - 2);
                    // 清理移除括号后可能出现的前后空格
                    first_non_space = formula_str.find_first_not_of(" \t\n\r");
                    if (first_non_space == std::string::npos) formula_str = "";
                    else {
                        size_t last_non_space = formula_str.find_last_not_of(" \t\n\r");
                        formula_str = formula_str.substr(first_non_space, last_non_space - first_non_space + 1);
                    }
                }
            }

            if (formula_str.empty()) { std::cerr << "警告: 行 " << line_num << ": 最终提取的公式为空，跳过。" << std::endl; continue; }

            // 解析 CNF 公式并转换为 CSE
            Clause cnf_clause = parse_cnf_formula(formula_str);
            if (cnf_clause.empty()) { std::cerr << "警告: 行 " << line_num << ": 解析出的子句为空，跳过。" << std::endl; continue; }

            int local_var_idx = 1; // 为每个子句重置局部变量计数器 (x<clause_id>1, x<clause_id>2, ...)
            std::unordered_map<std::string, std::string> local_variable_map; // 每个子句独立的变量映射
            std::string cse_clause_str; // 用于构建 CSE 格式的子句字符串

            // 遍历子句中的每个文字，将其转换为 CSE 格式
            for (size_t i = 0; i < cnf_clause.size(); ++i) {
                // 调用转换函数，传入 mapper, 局部变量映射, 子句 ID 和局部变量索引
                std::string lit_str = convert_literal_to_cse(cnf_clause[i], mapper, local_variable_map, clause_id_number, local_var_idx);

                cse_clause_str += lit_str;
                if (i < cnf_clause.size() - 1) {
                    cse_clause_str += "+"; // CSE 子句中的文字用 '+' 连接 (表示析取)
                }
            }

            // 将转换后的 CSE 子句写入输出文件，格式为 "[ID]Literal1+Literal2+..."
            cse_output_file << "[" << clause_id_number << "]" << cse_clause_str << std::endl;

        } catch (const std::exception& e) {
            // 捕获解析或转换过程中可能抛出的异常
            std::cerr << "错误: 处理行 " << line_num << " 失败。 原因: " << e.what() << std::endl;
            std::cerr << "  原始行内容: " << line << std::endl;
            success = false; // 标记发生了错误
        }
    }

    // 4. 关闭文件
    cnf_file.close();      // 关闭 CNF 输入文件
    cse_output_file.close(); // 关闭 CSE 输出文件

    return success;
}

// =====================================================================
// === 以下为新增内容 ===
// =====================================================================

// --- CSE 还原为原始格式的实现 ---

// 辅助函数：转义正则表达式中的特殊字符
std::string escape_regex_string(const std::string& s) {
    static const std::regex special_chars { R"([-[\]{}()*+?.,\^$|#\s])" };
    return std::regex_replace(s, special_chars, R"(\$&)");
}

// 辅助函数：去除字符串两端的空白字符
std::string trim_string(const std::string& str) {
    const std::string whitespace = " \t\n\r\f\v";
    size_t start = str.find_first_not_of(whitespace);
    if (start == std::string::npos) { // 字符串全是空白或为空
        return "";
    }
    size_t end = str.find_last_not_of(whitespace);
    return str.substr(start, end - start + 1);
}

// 全局映射表缓存，避免重复构建
static std::unordered_map<std::string, std::string> g_cse_to_original_cache;
static std::vector<std::string> g_sorted_cse_symbols_cache;
static bool g_cache_initialized = false;

/**
 * 初始化符号映射缓存
 * 1. 合并所有反向映射表（谓词、函数、常量）到统一缓存中
 * 2. 确保特殊符号"E"正确映射到等号"="
 * 3. 按符号长度降序排序，避免短符号误匹配长符号的问题
 * 4. 设置全局缓存标记，避免重复初始化
 */
void initialize_mapping_cache(const CseSymbolMapper& mapper) {
    if (g_cache_initialized) return;

    const auto& r_pred_map = mapper.getReversePredicateMap();
    const auto& r_func_map = mapper.getReverseFunctionMap();
    const auto& r_const_map = mapper.getReverseConstantMap();

    g_cse_to_original_cache.clear();
    g_sorted_cse_symbols_cache.clear();

    // 填充映射表
    for (const auto& p : r_func_map) {
        if (!p.first.empty()) g_cse_to_original_cache[p.first] = p.second;
    }
    for (const auto& p : r_const_map) {
        if (!p.first.empty()) g_cse_to_original_cache[p.first] = p.second;
    }
    for (const auto& p : r_pred_map) {
        if (!p.first.empty()) g_cse_to_original_cache[p.first] = p.second;
    }

    // 确保E映射到=
    g_cse_to_original_cache["E"] = "=";

    // 构建排序后的符号列表
    for (const auto& pair : g_cse_to_original_cache) {
        if (!pair.first.empty()) {
            g_sorted_cse_symbols_cache.push_back(pair.first);
        }
    }

    // 按长度降序排序
    std::sort(g_sorted_cse_symbols_cache.begin(), g_sorted_cse_symbols_cache.end(),
              [](const std::string& a, const std::string& b) {
                  return a.length() > b.length();
              });

    g_cache_initialized = true;
}

/**
 * 符号替换函数
 *
 * 该函数将子句字符串中的CSE符号替换为对应的原始符号。
 * 使用预构建的缓存映射表和词边界检查确保替换的准确性和效率。
 *
 * 处理流程：
 * 1. 初始化映射缓存（首次调用时）
 * 2. 按符号长度降序遍历，避免短符号误匹配长符号的问题
 * 3. 对每个CSE符号进行词边界检查，确保完整匹配
 * 4. 执行字符串替换，将CSE符号替换为原始符号
 */
std::string remap_symbols_in_clause(const std::string& clause_str,
                                    const CseSymbolMapper& mapper) {
    // 初始化缓存（只在第一次调用时执行）
    initialize_mapping_cache(mapper);

    if (g_cse_to_original_cache.empty()) return clause_str;

    std::string result_str = clause_str;

    // 使用缓存的排序符号列表
    for (const std::string& cse_symbol : g_sorted_cse_symbols_cache) {
        auto map_it = g_cse_to_original_cache.find(cse_symbol);
        if (map_it == g_cse_to_original_cache.end()) continue;

        const std::string& original_symbol = map_it->second;

        // 如果符号相同，跳过替换
        if (cse_symbol == original_symbol) continue;

        size_t pos = 0;

        // 预先计算符号长度
        const size_t cse_len = cse_symbol.length();
        const size_t orig_len = original_symbol.length();

        // 查找并替换所有出现的CSE符号
        while ((pos = result_str.find(cse_symbol, pos)) != std::string::npos) {
            // 检查词边界
            bool valid_prefix = (pos == 0) ||
                                (!std::isalnum(static_cast<unsigned char>(result_str[pos - 1])) &&
                                 result_str[pos - 1] != '_');

            size_t end_pos = pos + cse_len;
            bool valid_suffix = (end_pos >= result_str.length()) ||
                                (!std::isalnum(static_cast<unsigned char>(result_str[end_pos])) &&
                                 result_str[end_pos] != '_');

            if (valid_prefix && valid_suffix) {
                // 执行替换
                result_str.replace(pos, cse_len, original_symbol);
                pos += orig_len; // 跳过替换后的内容
            } else {
                pos += cse_len; // 跳过当前匹配
            }
        }
    }

    return result_str;
}

/**
 * 还原无参谓词
 * 该函数将CSE转换过程中为无参谓词添加的特殊常量参数移除，
 * 恢复原始的无参谓词形式。
 */
//std::string restore_nullary_predicates(const std::string& clause_str) {
//    std::string result = clause_str;
//    const std::string pattern = "(a99999)";
//    const std::string replacement = "()";
//
//    size_t pos = 0;
//    while ((pos = result.find(pattern, pos)) != std::string::npos) {
//        // 检查前面是否是有效的标识符字符
//        bool valid_prefix = false;
//        if (pos > 0) {
//            // 向前查找标识符的开始
//            size_t identifier_start = pos - 1;
//            while (identifier_start > 0 &&
//                   (std::isalnum(static_cast<unsigned char>(result[identifier_start - 1])) ||
//                    result[identifier_start - 1] == '_')) {
//                identifier_start--;
//            }
//
//            // 检查是否是有效的标识符
//            if (identifier_start < pos &&
//                (std::isalpha(static_cast<unsigned char>(result[identifier_start])) ||
//                 result[identifier_start] == '_')) {
//                valid_prefix = true;
//            }
//        }
//
//        if (valid_prefix) {
//            result.replace(pos, pattern.length(), replacement);
//            pos += replacement.length();
//        } else {
//            pos += pattern.length();
//        }
//    }
//
//    return result;
//}

/**
 * 提取顶层参数列表
 *
 * 该函数从参数字符串中提取顶层的参数，正确处理嵌套的括号结构。
 * 只在括号平衡为0时才将逗号识别为参数分隔符。
 *
 * 处理逻辑：
 * 1. 维护括号平衡计数器，跟踪嵌套层级
 * 2. 只有在顶层（平衡计数=0）时，逗号才被视为参数分隔符
 * 3. 对每个提取的参数去除前后空白字符
 * 4. 正确处理最后一个参数（没有后续逗号）
 */
std::vector<std::string> extract_top_level_arguments(const std::string& args_str_content_const) {
    std::string args_str_content = trim_string(args_str_content_const); // 首先清理整个参数字符串
    std::vector<std::string> arguments;
    if (args_str_content.empty()) {
        return arguments;
    }

    int balance = 0;
    size_t current_arg_start_idx = 0;

    for (size_t i = 0; i < args_str_content.length(); ++i) {
        if (args_str_content[i] == '(') {
            balance++;
        } else if (args_str_content[i] == ')') {
            balance--;
            if (balance < 0) {
            }
        } else if (args_str_content[i] == ',' && balance == 0) {
            arguments.push_back(trim_string(args_str_content.substr(current_arg_start_idx, i - current_arg_start_idx)));
            current_arg_start_idx = i + 1;
        }
    }
    // 添加最后一个（或唯一的）参数
    arguments.push_back(trim_string(args_str_content.substr(current_arg_start_idx)));
    return arguments;
}

/**
 * 将前缀等式表达式转换为中缀表达式
 *
 * 处理流程：
 * 1. 逐个字符扫描文本，查找等号'='
 * 2. 检查等号前是否有否定符号'~'
 * 3. 验证等号后是否跟有参数列表'(...)'
 * 4. 使用括号平衡算法找到匹配的右括号
 * 5. 提取并解析参数列表（必须恰好2个参数）
 * 6. 根据是否否定生成相应的中缀表达式
 */
std::string transform_equality_to_infix(const std::string& text) {
    std::string built_string;
    size_t current_search_pos = 0;

    while(current_search_pos < text.length()) {
        // 查找 '='
        size_t eq_pos = text.find('=', current_search_pos);
        if (eq_pos == std::string::npos) { // 未找到更多的 '='
            built_string += text.substr(current_search_pos);
            break;
        }
        // 将 '='之前的部分追加到结果字符串
        built_string += text.substr(current_search_pos, eq_pos - current_search_pos);

        bool is_negated = false;
        // 检查'='是否被否定
        if (!built_string.empty() && built_string.back() == '~') {
            is_negated = true;
            built_string.pop_back();
        } else if (built_string.length() >= 2 && built_string[built_string.length()-1] == ' ' && built_string[built_string.length()-2] == '~') {
            is_negated = true;
            built_string.resize(built_string.length() - 2); // 移除 "~ "
        }

        // 检查'='后面是否紧跟着参数列表的'('
        size_t open_paren_pos = eq_pos + 1;
        while(open_paren_pos < text.length() && std::isspace(text[open_paren_pos])) {
            open_paren_pos++; // 跳过'='和'('之间的空格
        }

        if (open_paren_pos < text.length() && text[open_paren_pos] == '(') {
            // 找到了 "=(", 尝试寻找匹配的 ')'
            int balance = 1; // 起始'('的计数
            size_t args_content_start = open_paren_pos + 1;
            size_t args_content_end = std::string::npos;
            size_t current_char_idx = args_content_start;

            while(current_char_idx < text.length()) {
                if (text[current_char_idx] == '(') {
                    balance++;
                } else if (text[current_char_idx] == ')') {
                    balance--;
                    if (balance == 0) { // 找到匹配的闭括号
                        args_content_end = current_char_idx;
                        break;
                    }
                }
                current_char_idx++;
            }

            if (args_content_end != std::string::npos) { // 成功找到匹配的括号
                std::string args_content = text.substr(args_content_start, args_content_end - args_content_start);
                std::vector<std::string> arguments = extract_top_level_arguments(args_content);

                if (arguments.size() == 2) {
                    // 成功解析出两个参数
                    if(is_negated) {
                        built_string += arguments[0] + "!=" + arguments[1];
                    } else {
                        built_string += arguments[0] + "=" + arguments[1];
                    }

                    current_search_pos = args_content_end + 1; // 从处理过的块之后继续搜索
                    continue; // 继续主循环
                }
            }
        }

        built_string += text[eq_pos];
        current_search_pos = eq_pos + 1;

    } // 结束 while 循环

    return built_string;
}

/**
 * 重命名子句体中的变量
 *
 * 该函数将CSE格式中的内部变量名转换为标准化形式。
 * 按照变量在子句中的出现顺序依次分配新的变量名。
 *
 * 重命名规则：
 * - 查找所有"x"开头后跟数字的变量（如x1231, x1232, x456等）
 * - 按照在子句中首次出现的顺序分配新名称：X1, X2, X3...
 * - 确保同一个原变量在整个子句中的所有出现都映射到同一个新变量
 *
 * 处理流程：
 * 1. 扫描整个字符串，识别所有CSE格式的变量
 * 2. 为每个新发现的变量分配标准化名称
 * 3. 按原变量名长度降序进行替换，避免短名称误匹配长名称
 * 4. 使用词边界检查确保完整匹配
 */
std::string rename_variables_in_clause_body(const std::string& body_str) {
    std::string current_body = body_str;
    std::map<std::string, std::string> old_to_new_map;
    int next_var_idx = 1;

    std::vector<std::string> found_vars_in_order; // 用于确保替换顺序与发现顺序一致
    for (size_t i = 0; (i = current_body.find('x', i)) != std::string::npos; ) {
        if (i + 1 < current_body.length() && std::isdigit(static_cast<unsigned char>(current_body[i+1]))) {
            size_t var_start_idx = i;
            size_t var_end_idx = i + 1;
            while (var_end_idx < current_body.length() && std::isdigit(static_cast<unsigned char>(current_body[var_end_idx]))) {
                var_end_idx++;
            }
            std::string original_var = current_body.substr(var_start_idx, var_end_idx - var_start_idx);

            bool prefix_ok = (var_start_idx == 0 || !std::isalnum(static_cast<unsigned char>(current_body[var_start_idx - 1])));
            bool suffix_ok = (var_end_idx == current_body.length() || !std::isalnum(static_cast<unsigned char>(current_body[var_end_idx])));

            if (prefix_ok && suffix_ok) {
                bool found_in_map = false;
                for(const auto& pair : old_to_new_map) {
                    if (pair.first == original_var) {
                        found_in_map = true;
                        break;
                    }
                }
                if (!found_in_map) {
                    std::string new_mapping = "X" + std::to_string(next_var_idx++);
                    old_to_new_map[original_var] = new_mapping;
                    found_vars_in_order.push_back(original_var);
                }
            }
            i = var_end_idx;
        } else {
            i++;
        }
    }

    if (old_to_new_map.empty()) {
        return current_body;
    }

    // 按原始变量名长度降序排序进行替换
    std::vector<std::pair<std::string, std::string>> sorted_mappings_for_replace(old_to_new_map.begin(), old_to_new_map.end());
    std::sort(sorted_mappings_for_replace.begin(), sorted_mappings_for_replace.end(),
              [](const std::pair<std::string, std::string>& a, const std::pair<std::string, std::string>& b) -> bool {
                  return a.first.length() > b.first.length();
              });

    for (const auto& pair_to_replace : sorted_mappings_for_replace) {
        const std::string& old_var = pair_to_replace.first;
        const std::string& new_var = pair_to_replace.second;
        size_t pos = 0;
        while ((pos = current_body.find(old_var, pos)) != std::string::npos) {
            bool prefix_ok = (pos == 0 || !std::isalnum(static_cast<unsigned char>(current_body[pos - 1])));
            bool suffix_ok = ((pos + old_var.length()) == current_body.length() || !std::isalnum(static_cast<unsigned char>(current_body[pos + old_var.length()])));

            if (prefix_ok && suffix_ok) {
                current_body.replace(pos, old_var.length(), new_var);
                pos += new_var.length();
            } else {
                pos += old_var.length();
            }
        }
    }

    return current_body;
}

/**
 * 处理单个CNF语句块
 *
 * 该函数处理单个完整的CNF语句，执行符号还原、
 * 格式转换和变量重命名等操作，将其从CSE格式还原为原始格式。
 *
 * 处理步骤：
 * 1. 验证CNF语句格式：必须以"cnf("开头，以")."结尾
 * 2. 解析CNF语句的三个部分：ID、类型、公式体
 * 3. 标准化ID格式：纯数字ID添加"i_0_"前缀
 * 4. 统一类型字段为"plain"
 * 5. 还原公式体中的符号和变量
 * 6. 转换等式表达式为中缀形式
 * 7. 为公式体添加外层括号
 */
std::string process_single_cnf_block(const std::string& cnf_block,
                                     const CseSymbolMapper& mapper,
                                     const std::vector<unsigned int>& goalIds) {
    std::string trimmed_block = trim_string(cnf_block);
    if (trimmed_block.rfind("cnf(", 0) == 0 && trimmed_block.length() > 5 &&
        trimmed_block.rfind(").", trimmed_block.length() - 2) == trimmed_block.length() - 2) {

        // 提取 cnf(...) 内部的内容
        std::string content = trimmed_block.substr(4, trimmed_block.length() - 4 - 2);

        size_t first_comma = content.find(',');
        if (first_comma == std::string::npos) return cnf_block;

        size_t second_comma = content.find(',', first_comma + 1);
        if (second_comma == std::string::npos) return cnf_block;

        std::string id_str = trim_string(content.substr(0, first_comma));
        std::string type_str = trim_string(content.substr(first_comma + 1, second_comma - (first_comma + 1)));
        std::string clause_body_str = content.substr(second_comma + 1);

        // 处理ID格式：如果是纯数字，添加 i_0_ 前缀
        bool is_all_digits = !id_str.empty();
        for (char c : id_str) {
            if (!std::isdigit(static_cast<unsigned char>(c))) {
                is_all_digits = false;
                break;
            }
        }

        unsigned int clause_id = std::stoul(id_str);

        if (is_all_digits) {
            id_str = "c_0_" + id_str;
        }

        // 确定子句类型
        type_str = "plain";
        if (std::find(goalIds.begin(), goalIds.end(), clause_id) != goalIds.end()) {
            type_str = "negated_conjecture";
        }

        // 第一步: CSE符号还原和变量重命名
        std::string final_clause_body = restore_clause_body_optimized(clause_body_str, mapper);

        // 第二步: 等式转换 E(a,b)/=(a,b) -> a=b
        final_clause_body = transform_equality_to_infix(final_clause_body);

        // 为公式部分添加外层括号
        final_clause_body = "(" + final_clause_body + ")";

        return "cnf(" + id_str + ", " + type_str + ", " + final_clause_body + ").";
    }
    return cnf_block;
}

/**
 * 将CSE格式行转换为CNF格式行
 *
 * 该函数将纯CSE格式的行转换为标准CNF格式，便于后续的还原处理。
 *
 * 处理流程：
 * 1. 检查是否为CSE格式（以'['开头）
 * 2. 提取方括号中的ID部分
 * 3. 提取方括号后的公式部分
 * 4. 将公式中的'+'替换为'|'
 * 5. 构造标准CNF格式字符串
 *
 */
std::string convert_cse_line_to_cnf(const std::string& cse_line, const std::vector<unsigned int>& goalIds) {
    std::string trimmed_line = trim_string(cse_line);
    if (trimmed_line.empty()) {
        return trimmed_line;
    }

    // 检查是否为CSE格式：[ID]formula
    if (trimmed_line[0] != '[') {
        return trimmed_line; // 不是CSE格式，原样返回
    }

    // 查找右括号
    size_t right_bracket = trimmed_line.find(']');
    if (right_bracket == std::string::npos) {
        return trimmed_line; // 格式错误，原样返回
    }

    // 提取ID
    std::string id_str = trimmed_line.substr(1, right_bracket - 1);

    // 提取公式部分
    std::string formula_part = trimmed_line.substr(right_bracket + 1);

    // 将 '+' 替换为 '|'（CSE中用+表示析取，CNF中用|表示）
    std::string cnf_formula = formula_part;
    size_t pos = 0;
    while ((pos = cnf_formula.find('+', pos)) != std::string::npos) {
        cnf_formula.replace(pos, 1, "|");
        pos += 1; // 跳过 "|"
    }

    // 确定子句类型
    std::string clause_type = "plain";
    unsigned int clause_id = std::stoul(id_str);
    if (std::find(goalIds.begin(), goalIds.end(), clause_id) != goalIds.end()) {
        clause_type = "negated_conjecture";
    }

    // 构造CNF格式
    return "cnf(" + id_str + ",axiom," + cnf_formula + ").";
}

/**
 * CSE到原始格式的主转换函数
 *
 * 该函数是CSE还原功能的核心入口点，负责读取CSE格式或CNF格式的文件，
 * 并将其还原为原始的、易读的CNF格式。支持处理跨行的CNF语句和CSE格式文件。
 *
 *
 * 处理流程：
 * 1. 自动检测文件格式（首行是否以'['开头判断是否为CSE格式）
 * 2. CSE格式：先转换为CNF格式再进行还原
 * 3. CNF格式：直接进行符号还原和格式优化
 * 4. 处理跨行的CNF语句，使用语句累加器
 * 5. 保留非CNF行的原始格式（注释、空行等）
 *
 */
bool convert_cse_to_cnf(const std::string& input_cse_cnf_path,
                             const std::string& output_original_path,
                             const CseSymbolMapper& mapper,
                             const std::vector<unsigned int>& goalIds) {
    std::ifstream infile(input_cse_cnf_path);
    if (!infile.is_open()) {
        std::cerr << "错误: 无法打开输入文件 '" << input_cse_cnf_path << "'" << std::endl;
        return false;
    }

    std::ofstream outfile(output_original_path);
    if (!outfile.is_open()) {
        std::cerr << "错误: 无法打开输出文件 '" << output_original_path << "'" << std::endl;
        infile.close();
        return false;
    }

    std::string line;
    std::string cnf_accumulator;
    bool in_cnf_statement = false;
    int line_num = 0;
    bool is_pure_cse_format = false; // 标记是否为纯CSE格式文件

    // 首先检测文件格式
    std::string first_line;
    if (std::getline(infile, first_line)) {
        std::string trimmed_first = trim_string(first_line);
        if (!trimmed_first.empty() && trimmed_first[0] == '[') {
            is_pure_cse_format = true;
            std::cout << "#检测到纯CSE格式文件，将先转换为CNF格式再进行还原..." << std::endl;
        }
        // 重置文件指针到开头
        infile.clear();
        infile.seekg(0, std::ios::beg);
    }

    while (std::getline(infile, line)) {
        line_num++;

        // 如果是纯CSE格式，先转换为CNF格式
        if (is_pure_cse_format) {
            std::string converted_line = convert_cse_line_to_cnf(line, goalIds);
            line = converted_line;
        }

        std::string trimmed_line = trim_string(line);

        if (trimmed_line.empty() && !in_cnf_statement) { // 完全空行且不在cnf语句中
            outfile << std::endl; // 保留空行格式
            continue;
        }

        if (trimmed_line.rfind("cnf(", 0) == 0) { // 以 "cnf(" 开头
            if (in_cnf_statement) { // 如果已经在处理一个cnf语句，说明上一个没有正确结束
                outfile << process_single_cnf_block(cnf_accumulator, mapper, goalIds) << std::endl;
            }
            cnf_accumulator = line; // 开始新的累加
            in_cnf_statement = true;
        } else if (in_cnf_statement) {
            cnf_accumulator += "\n" + line; // 在累加器中用 \n 代表换行，原始行内容不变
        } else { // 不在 cnf 语句中，也不是 cnf 语句的开头
            outfile << line << std::endl; // 原样输出
        }

        if (in_cnf_statement) {
            // 是否包含完整的 cnf 语句
            std::string temp_check_str = cnf_accumulator;
            size_t n_pos;
            while((n_pos = temp_check_str.find("\\n")) != std::string::npos) {
                temp_check_str.replace(n_pos, 2, " "); // 用空格替换，以模拟单行进行检查
            }
            std::string trimmed_accumulator_for_check = trim_string(temp_check_str);

            if (trimmed_accumulator_for_check.length() > 2 && trimmed_accumulator_for_check.rfind(").", trimmed_accumulator_for_check.length() - 2) == trimmed_accumulator_for_check.length() - 2) {
                bool stuff_after_closing_paren_dot = false;

                std::string last_physical_line_in_accumulator;
                size_t last_newline_pos = cnf_accumulator.rfind("\\n");
                if (last_newline_pos != std::string::npos) {
                    last_physical_line_in_accumulator = cnf_accumulator.substr(last_newline_pos + 2);
                } else {
                    last_physical_line_in_accumulator = cnf_accumulator;
                }

                size_t dot_in_last_physical_line = last_physical_line_in_accumulator.rfind(").");
                if (dot_in_last_physical_line != std::string::npos && (dot_in_last_physical_line + 2 < last_physical_line_in_accumulator.length())) {
                    std::string tail = last_physical_line_in_accumulator.substr(dot_in_last_physical_line + 2);
                    if (!trim_string(tail).empty()) {
                        stuff_after_closing_paren_dot = true;
                    }
                }

                if (!stuff_after_closing_paren_dot) {
                    // 预读下一行
                    std::string next_line_peek;
                    bool eof_reached = false;
                    if (infile.peek() != EOF) {
                        // 存当前位置，读取，然后恢复
                        std::streampos current_pos = infile.tellg();
                        if (std::getline(infile, next_line_peek)) {
                            infile.seekg(current_pos); // 恢复到预读前的位置
                        } else {
                            eof_reached = true; // 到达文件末尾
                        }
                    } else {
                        eof_reached = true;
                    }

                    std::string trimmed_next_line_peek = trim_string(next_line_peek);
                    if (eof_reached || trimmed_next_line_peek.rfind("cnf(", 0) == 0 || trimmed_next_line_peek.rfind("%", 0) == 0 || trimmed_next_line_peek.empty()) {

                        std::string final_cnf_block_to_process = cnf_accumulator;
                        while((n_pos = final_cnf_block_to_process.find("\\n")) != std::string::npos) {
                            final_cnf_block_to_process.replace(n_pos, 2, "\n");
                        }
                        outfile << process_single_cnf_block(final_cnf_block_to_process, mapper, goalIds) << std::endl;
                        cnf_accumulator = "";
                        in_cnf_statement = false;
                    }
                }
            }
        }
    }

    // 文件处理完毕后，如果累加器中还有内容，处理它
    if (in_cnf_statement && !cnf_accumulator.empty()) {
        std::string final_cnf_block_to_process = cnf_accumulator;
        size_t n_pos;
        while((n_pos = final_cnf_block_to_process.find("\\n")) != std::string::npos) {
            final_cnf_block_to_process.replace(n_pos, 2, "\n");
        }
        outfile << process_single_cnf_block(final_cnf_block_to_process, mapper, goalIds) << std::endl;
    }

    infile.close();
    outfile.close();

    if (is_pure_cse_format) {
        std::cout << "#纯CSE格式文件转换并还原完成，输出文件已生成: " << output_original_path << std::endl;
    } else {
        std::cout << "#CNF格式文件还原完成，输出文件已生成: " << output_original_path << std::endl;
    }
    return true;
}

/**
 * 子句还原的内部优化函数
 *
 * 该函数在单次扫描中完成所有符号转换和变量重命名操作。
 *
 * 处理的转换类型：
 * 1. CSE符号还原：P1->原始谓词名, f1->原始函数名, a1->原始常量名
 * 2. 变量重命名：x1231->X1, x1232->X2 (按出现顺序分配)
 * 3. 无参谓词还原：移除特殊常量参数(a99999)
 */
std::string restore_clause_body_optimized_internal(const std::string& clause_body_str,
                                                   const CseSymbolMapper& mapper,
                                                   std::unordered_map<std::string, std::string>& local_var_rename_cache,
                                                   int& local_var_counter) {
    if (g_cse_to_original_cache.empty()) return clause_body_str;

    std::string result;
    result.reserve(clause_body_str.length() * 2); // 预分配内存

    size_t i = 0;
    const size_t len = clause_body_str.length();

    while (i < len) {
        char current_char = clause_body_str[i];

        // 处理标识符（可能是CSE符号或变量）
        if (std::isalpha(current_char) || current_char == '_') {
            size_t identifier_start = i;

            // 读取完整标识符
            while (i < len && (std::isalnum(clause_body_str[i]) || clause_body_str[i] == '_')) {
                i++;
            }

            std::string identifier = clause_body_str.substr(identifier_start, i - identifier_start);

            // 步骤1: 检查是否是CSE符号需要映射
            auto cse_it = g_cse_to_original_cache.find(identifier);
            if (cse_it != g_cse_to_original_cache.end()) {
                identifier = cse_it->second; // 替换为原始符号
            }

            // 步骤3: 检查是否是需要重命名的变量 (x开头的数字变量)
            if (identifier[0] == 'x' && identifier.length() > 1 && std::isdigit(identifier[1])) {
                // 简化变量重命名：x1231 -> X1, x1232 -> X2 等
                auto var_it = local_var_rename_cache.find(identifier);
                if (var_it == local_var_rename_cache.end()) {
                    std::string new_var = "X" + std::to_string(local_var_counter++);
                    local_var_rename_cache[identifier] = new_var;
                    identifier = new_var;
                } else {
                    identifier = var_it->second;
                }
            }

            result += identifier;
            continue;
        }

        // 步骤2: 处理无参谓词还原 (a99999) -> 移除
//        if (current_char == '(' && i + 6 < len &&
//            clause_body_str.substr(i, 7) == "(a99999") {
//            // 查找匹配的右括号
//            size_t close_paren_pos = i + 7;
//            while (close_paren_pos < len && clause_body_str[close_paren_pos] != ')') {
//                close_paren_pos++;
//            }
//            if (close_paren_pos < len && clause_body_str[close_paren_pos] == ')') {
//                i = close_paren_pos + 1;
//                continue;
//            }
//        }
        result += current_char;
        i++;
    }

    return result;
}

/**
 * 高效的子句体还原函数
 *
 * 该函数是子句体还原功能的对外统一接口，负责初始化必要的数据结构
 * 并调用内部优化函数完成实际的还原工作。
 *
 * 主要职责：
 * 1. 初始化全局符号映射缓存（首次调用时）
 * 2. 为每个子句创建独立的变量重命名上下文
 * 3. 调用内部优化函数执行实际转换
 *
 */
std::string restore_clause_body_optimized(const std::string& clause_body_str,
                                          const CseSymbolMapper& mapper) {
    // 初始化映射缓存
    initialize_mapping_cache(mapper);

    if (g_cse_to_original_cache.empty()) return clause_body_str;

    // 为每个子句创建独立的变量重命名映射
    std::unordered_map<std::string, std::string> local_var_rename_cache;
    int local_var_counter = 1;

    return restore_clause_body_optimized_internal(clause_body_str, mapper, local_var_rename_cache, local_var_counter);
}

// 将单个CSE格式的子句转换为CNF格式的子句
std::string convert_single_cse_clause_to_cnf(const std::string& cse_line,
                                             const CseSymbolMapper& mapper,
                                             const std::vector<unsigned int>& goalIds) {
    std::string trimmed_line = trim_string(cse_line);
    if (trimmed_line.empty()) {
        return "";
    }

    // 检查是否为CSE格式：[ID]formula
    if (trimmed_line[0] != '[') {
        return trimmed_line; // 不是CSE格式，原样返回
    }

    // 查找右括号
    size_t right_bracket = trimmed_line.find(']');
    if (right_bracket == std::string::npos) {
        return trimmed_line; // 格式错误，原样返回
    }

    // 提取ID
    std::string id_str = trimmed_line.substr(1, right_bracket - 1);

    // 提取公式部分
    std::string formula_part = trimmed_line.substr(right_bracket + 1);
    if (formula_part.empty()) {
        return "cnf(" + id_str + ", plain, ()).";
    }

    // 还原公式体中的符号和变量
    std::string restored_formula = restore_clause_body_optimized(formula_part, mapper);

    // 将 '+' 替换为 '|'（CSE中用+表示析取，CNF中用|表示）
    std::string cnf_formula;
    size_t start = 0, end = 0;
    while ((end = restored_formula.find('+', start)) != std::string::npos) {
        cnf_formula += restored_formula.substr(start, end - start) + "|";
        start = end + 1;
    }
    cnf_formula += restored_formula.substr(start);

    // 处理等式表达式（前缀转中缀）
    cnf_formula = transform_equality_to_infix(cnf_formula);

    // 添加外层括号
    cnf_formula = "(" + cnf_formula + ")";

    // 确定子句类型
    std::string clause_type = "plain";
    try {
        unsigned int clause_id = std::stoul(id_str);
        if (std::find(goalIds.begin(), goalIds.end(), clause_id) != goalIds.end()) {
            clause_type = "negated_conjecture";
        }
    } catch (...) {
        // 如果ID转换失败，保持默认类型
    }

    // 处理ID格式：如果是纯数字，添加i_0_前缀
    bool is_all_digits = !id_str.empty();
    for (char c : id_str) {
        if (!std::isdigit(static_cast<unsigned char>(c))) {
            is_all_digits = false;
            break;
        }
    }
    if (is_all_digits) {
        id_str = "c_0_" + id_str;
    }

    // 构造CNF格式
    return "cnf(" + id_str + ", " + clause_type + ", " + cnf_formula + ").";
}

