/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

/* 
 * File:   AddEqnAxioms.h
 * Author: ars01
 *
 * Created on 2025年6月11日, 下午8:16
 */

#ifndef ADDEQNAXIOMS_H
#define ADDEQNAXIOMS_H

#include <fstream>
#include <vector>
#include <set>
#include <map>
#include <algorithm>
#include <stdexcept>
#include <string>
#include <sstream>
#include <cctype>
#include <iostream>
#include <unordered_set>

using namespace std;

// 路径处理函数
string get_output_path(const string& input_path, bool has_equality) {
    size_t last_slash = input_path.find_last_of("/\\");
    string dir = (last_slash != string::npos) ? input_path.substr(0, last_slash + 1) : "";
    string file = (last_slash != string::npos) ? input_path.substr(last_slash + 1) : input_path;

    size_t dot_pos = file.rfind('.');
    string stem = (dot_pos != string::npos) ? file.substr(0, dot_pos) : file;

    string suffix = has_equality ? "_short+eq_rstfp.cse" : "_short+noeq.cse";
    return dir + stem + suffix;
}

// 符号信息结构体
struct SymbolInfo {
    bool has_equality = false;
    set<string> functions;
    map<string, int> func_arity;
    set<string> predicates;
    map<string, int> pred_arity;
    int max_clause_num = 0;
};

// 检查是否是有效的CNF子句
bool is_valid_clause(const string& line) {
    size_t pos = 0;

    // 检查起始方括号
    if (line.empty() || line[pos++] != '[') return false;

    // 提取数字编号
    bool has_digit = false;
    while (pos < line.size() && isdigit(line[pos])) {
        has_digit = true;
        pos++;
    }
    if (!has_digit) return false;

    // 检查闭合方括号
    if (pos >= line.size() || line[pos] != ']') return false;
    pos++;

    // 剩余部分至少需要有一个字符
    return pos < line.size();
}

// 预处理函数
string preprocess_line(string line) {
    // 删除所有控制字符和空格
    line.erase(remove_if(line.begin(), line.end(),
                         [](unsigned char c){ return c < 32 || isspace(c); }), line.end());
    return line;
}

// 符号分析函数
void analyze_symbols(const string& clause, SymbolInfo& info) {
    size_t bracket_end = clause.find(']');
    if (bracket_end == string::npos) return;

    size_t pos = bracket_end + 1;
    while (pos < clause.size()) {
        // 查找函数/谓词符号
        if (isalpha(clause[pos])) {
            size_t start = pos;
            while (pos < clause.size() && (isalnum(clause[pos]) || clause[pos] == '_')) {
                pos++;
            }
            string symbol = clause.substr(start, pos - start);

            // 检查是否是等词E
            if (symbol == "E") {
                info.has_equality = true;
                // 等词始终是二元谓词
                info.predicates.insert("E");
                info.pred_arity["E"] = 2;
                continue;
            }

            // 检查符号后是否有括号
            if (pos < clause.size() && clause[pos] == '(') {
                size_t start_pos = pos;
                int paren_count = 1;
                vector<string> params;
                string current_param = "";

                pos++; // 跳过 '('
                while (pos < clause.size() && paren_count > 0) {
                    if (clause[pos] == '(') paren_count++;
                    else if (clause[pos] == ')') paren_count--;

                    if (paren_count == 1) {
                        if (clause[pos] == ',') {
                            params.push_back(current_param);
                            current_param = "";
                        } else if (clause[pos] != ')') {
                            current_param += clause[pos];
                        }
                    }

                    pos++;
                }

                if (!current_param.empty()) {
                    params.push_back(current_param);
                }

                int arity = params.size();

                // 确定符号类型（大写开头是谓词，小写是函数）
                if (isupper(symbol[0])) {
                    info.predicates.insert(symbol);
                    info.pred_arity[symbol] = arity;
                } else {
                    info.functions.insert(symbol);
                    info.func_arity[symbol] = arity;
                }
            }
        } else {
            pos++;
        }
    }
}

// 生成带指定格式变量的等词公理
vector<string> generate_equality_axioms(const SymbolInfo& info) {
    vector<string> axioms;
    if (!info.has_equality) return axioms;

    int axiom_num = info.max_clause_num + 1;

    // 基础等词公理（采用xNM格式）
    axioms.push_back("[" + to_string(axiom_num) + "]E(x" + to_string(axiom_num) + "1,x" + to_string(axiom_num) + "1)");
    axioms.push_back("[" + to_string(axiom_num+1) + "]E(x" + to_string(axiom_num+1) + "2,x" + to_string(axiom_num+1) + "1)+~E(x" + to_string(axiom_num+1) + "1,x" + to_string(axiom_num+1) + "2)");
    axioms.push_back("[" + to_string(axiom_num+2) + "]E(x" + to_string(axiom_num+2) + "1,x" + to_string(axiom_num+2) + "3)+~E(x" + to_string(axiom_num+2) + "1,x" + to_string(axiom_num+2) + "2)+~E(x" + to_string(axiom_num+2) + "2,x" + to_string(axiom_num+2) + "3)");
    axiom_num += 3;  // 增加了3条公理

    // 生成函数置换公理（采用xNM格式）
    for (const auto& func : info.functions) {
        int arity = info.func_arity.at(func);

        // 为每个参数位置生成独立的置换公理
        for (int pos = 0; pos < arity; pos++) {
            // 构建变量名 (xNM格式)
            vector<string> vars;
            vars.push_back("x" + to_string(axiom_num) + "1"); // 等词左边
            vars.push_back("x" + to_string(axiom_num) + "2"); // 等词右边

            // 添加其他变量
            for (int i = 0; i < arity; i++) {
                if (i != pos) {
                    vars.push_back("x" + to_string(axiom_num) + to_string(3 + i));
                }
            }

            string clause = "[" + to_string(axiom_num) + "]~E(" + vars[0] + "," + vars[1] + ")+E(" + func + "(";

            // 构建第一个函数调用
            vector<string> args1;
            for (int i = 0; i < arity; i++) {
                if (i == pos) {
                    args1.push_back(vars[0]);
                } else {
                    // 找到非置换位置的变量索引
                    int var_idx = 2;
                    for (int j = 0; j < i; j++) {
                        if (j == pos) continue;
                        var_idx++;
                    }
                    args1.push_back(vars[var_idx]);
                }
                clause += args1.back();
                if (i < arity - 1) clause += ",";
            }

            clause += ")," + func + "(";

            // 构建第二个函数调用
            for (int i = 0; i < arity; i++) {
                if (i == pos) {
                    clause += vars[1];
                } else {
                    // 找到非置换位置的变量索引
                    int var_idx = 2;
                    for (int j = 0; j < i; j++) {
                        if (j == pos) continue;
                        var_idx++;
                    }
                    clause += vars[var_idx];
                }
                if (i < arity - 1) clause += ",";
            }

            clause += "))";

            axioms.push_back(clause);
            axiom_num++;
        }
    }

    // 生成谓词置换公理（采用xNM格式）
    for (const auto& pred : info.predicates) {
        // 跳过等词自身
        if (pred == "E") continue;

        int arity = info.pred_arity.at(pred);

        // 为每个参数位置生成独立的置换公理
        for (int pos = 0; pos < arity; pos++) {
            // 构建变量名 (xNM格式)
            vector<string> vars;
            vars.push_back("x" + to_string(axiom_num) + "1"); // 等词左边
            vars.push_back("x" + to_string(axiom_num) + "2"); // 等词右边

            // 添加其他变量
            for (int i = 0; i < arity; i++) {
                if (i != pos) {
                    vars.push_back("x" + to_string(axiom_num) + to_string(3 + i));
                }
            }

            string clause = "[" + to_string(axiom_num) + "]~E(" + vars[0] + "," + vars[1] + ")+~" + pred + "(";

            // 构建原始谓词调用
            for (int i = 0; i < arity; i++) {
                if (i == pos) {
                    clause += vars[0];
                } else {
                    // 找到非置换位置的变量索引
                    int var_idx = 2;
                    for (int j = 0; j < i; j++) {
                        if (j == pos) continue;
                        var_idx++;
                    }
                    clause += vars[var_idx];
                }
                if (i < arity - 1) clause += ",";
            }

            clause += ")+" + pred + "(";

            // 构建替换后谓词调用
            for (int i = 0; i < arity; i++) {
                if (i == pos) {
                    clause += vars[1];
                } else {
                    // 找到非置换位置的变量索引
                    int var_idx = 2;
                    for (int j = 0; j < i; j++) {
                        if (j == pos) continue;
                        var_idx++;
                    }
                    clause += vars[var_idx];
                }
                if (i < arity - 1) clause += ",";
            }

            clause += ")";

            axioms.push_back(clause);
            axiom_num++;
        }
    }

    return axioms;
}

// 主处理函数
string convert_cnf_manual(const string& input_path) {
    ifstream in(input_path);
    if (!in) throw runtime_error("无法打开输入文件");

    vector<string> clauses;
    string line;
    while (getline(in, line)) {
        line = preprocess_line(line);
        if (!line.empty() && is_valid_clause(line)) {
            clauses.push_back(line);
        }
    }

    SymbolInfo info;
    for (const auto& clause : clauses) {
        // 提取子句编号
        size_t end_bracket = clause.find(']');
        if (end_bracket != string::npos) {
            try {
                int num = stoi(clause.substr(1, end_bracket - 1));
                info.max_clause_num = max(info.max_clause_num, num);
            } catch (const exception&) {
                // 忽略解析错误的编号
            }
        }

        // 符号分析
        analyze_symbols(clause, info);
    }

    string output_path = get_output_path(input_path, info.has_equality);
    ofstream out(output_path);
    for (const auto& c : clauses) out << c << "\n";

    if (info.has_equality) {
        vector<string> axioms = generate_equality_axioms(info);
        if (!axioms.empty()) {
            out << "%EqnAxiom\n";
            for (const auto& axiom : axioms) {
                out << axiom << "\n";
            }
        }
    }

    //cout << "添加等词公理完成: " << output_path << endl;
    return output_path;
}

#endif /* ADDEQNAXIOMS_H */

